<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Adhoc task for GRoup Enrollment Report
 *
 * @package     report_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs\task;

use block_ned_teacher_tools\deadline_manager;
use report_ghs\helper;
use stdClass;

defined('MOODLE_INTERNAL') || die();

global $CFG;
require_once($CFG->dirroot . '/report/ghs/lib.php');
require_once($CFG->dirroot.'/local/kica/lib.php');
require_once($CFG->dirroot.'/grade/grading/lib.php');
require_once($CFG->dirroot.'/lib/gradelib.php');


/**
 * Class adhoc_ghs_group_enrollment_report
 * @package report_ghs\task
 */
class adhoc_ghs_group_enrollment_report extends base_ghs_adhoc_task {

    /**
     * @return bool
     */
    public function execute() {
        global $CFG, $DB;

        $cd = $this->get_custom_data();

        self::update_report($cd->userid ?? null);

        return true;
    }

    /**
     * @return mixed
     * @throws \dml_exception
     */
    public static function get_task() {
        global $DB;

        $classname = self::class;
        if ($task = $DB->get_record('task_adhoc', ['classname' => '\\'.$classname])) {
            $task->customdata = json_decode($task->customdata);
        }

        return $task;
    }

    /**
     * @param null $userid
     * @return bool
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function update_report($userid=null) {
        global $CFG, $DB;

        $cfg = get_config('report_ghs');

        //$DB->execute("TRUNCATE TABLE {report_ghs_group_enrollment}");
        $DB->delete_records('report_ghs_group_enrollment');

        $contextsystem = \context_system::instance();

        $courselist = \report_ghs\helper::get_report_courses();

        list($insql, $params) = $DB->get_in_or_equal($courselist);

        $otrole = $DB->get_record('role', array('shortname' => 'online-teacher'));
        $ctrole = $DB->get_record('role', array('shortname' => 'classroom-teacher'));
        $handler  = \core_course\customfield\course_handler::create();

        // Groups.
        $sql = "SELECT * FROM {groups} g WHERE g.courseid {$insql} ORDER BY g.name ASC ";

        $sqlcohort = "SELECT cm.cohortid, COUNT(1) numofmembers 
                        FROM {cohort_members} cm 
                  INNER JOIN {cohort} c
                          ON cm.cohortid = c.id
                       WHERE c.name NOT LIKE '*%'
                         AND cm.userid IN ((SELECT gm.userid FROM {groups_members} gm WHERE gm.groupid = ?)) 
                    GROUP BY cm.cohortid
                    ORDER BY numofmembers DESC";

        $rs = $DB->get_recordset_sql($sql, $params);

        foreach ($rs as $group) {
            $coursecontext = \context_course::instance($group->courseid);

            $activestudents = get_enrolled_users($coursecontext, 'mod/assign:submit', $group->id, 'u.*', 'u.id', 0, 0, true);
            $allstudents = get_enrolled_users($coursecontext, 'mod/assign:submit', $group->id, 'u.*', 'u.id', 0, 0, false);

            foreach ($activestudents as $index => $activestudent) {
                if ($activestudent->suspended == 1) {
                    unset($activestudents[$index]);
                }
            }

            $data = new \stdClass();
            $data->courseid = $group->courseid;
            $data->groupid = $group->id;
             if ($cohorts = $DB->get_records_sql($sqlcohort, [$group->id])) {
                 $data->cohortid = (reset($cohorts))->cohortid;
             }

            $data->activestudents = count($activestudents);
            $data->suspendedstudents = count($allstudents) - $data->activestudents;
            if ($ctusers = helper::get_role_users($ctrole->id, $coursecontext, ((isset($group)) ? $group->id : 0) )) {
                $data->ctid = (count($ctusers) > 1) ? -1 : $ctusers[0]->id;
            }
            if ($otusers = helper::get_role_users($otrole->id, $coursecontext, ((isset($group)) ? $group->id : 0) )) {
                $data->otid = (count($otusers) > 1) ? -1 : $otusers[0]->id;
            }

            if ($group->schedule) {
                $deadlinemanager = new \block_ned_teacher_tools\deadline_manager($group->courseid);
                if ($missedschedule = $deadlinemanager->has_missed_schedule($group->id)) {
                    $data->dmstatus = get_string('incomplete', 'block_ned_teacher_tools');
                } else {
                    $data->dmstatus = get_string('complete');
                }
            }

            // $data->dmdateconflicts = self::count_dm_conflict($group);
            // $data->enrolldateconflicts = self::count_enrollment_conflict($group, $activestudents);

            // Custom course data.
            $customdata = $handler->export_instance_data_object($data->courseid, true);
            $customfields = ['subject', 'grade', 'department', 'moe_code'];
            foreach ($customfields as $customfield) {
                $data->{$customfield} = (isset($customdata->{$customfield})) ? $customdata->{$customfield} : '';
            }

            $data->timecreated = time();
            $DB->insert_record('report_ghs_group_enrollment', $data);
        }

        $rs->close();

        set_config('groupenrollmentlastupdate', time(), 'report_ghs');

        if (empty($CFG->noemailever) && $userid) {
            $site = get_site();
            $from = \core_user::get_support_user();
            $from->firstname = $site->fullname;
            $from->lastname = "";

            $to = $DB->get_record('user', array('id' => $userid));
            $to->mailformat = 1;

            email_to_user($to, $from, get_string('ghsgroupenrollment', 'report_ghs'), '', "TASK HAS BEEN EXECUTED!");
        }

        return true;
    }

    /**
     * @param $group
     * @return int
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function count_dm_conflict($group) {
        $counter = 0;

        if ($group->startdate == 0 && $group->enddate == 0) {
            return $counter;
        }
        $deadlinemanager = new deadline_manager($group->courseid);

        $modinfo = get_fast_modinfo($group->courseid);
        $activities = $modinfo->get_cms();

        foreach ($activities as $mod) {
            if (!$deadlinemanager->is_enabled_activity($mod->id)) {
                continue;
            }

            $classname = '\block_ned_teacher_tools\mod\deadline_manager_'. $mod->modname;
            if (!class_exists($classname)) {
                continue;
            }
            $module = new $classname($mod);

            if (!$deadline = $module->get_group_override_date($group->id)) {
                $deadline = $module->get_default_deadline();
            }

            if (empty($deadline)) {
                continue;
            }

            // Conflict
            if ($group->startdate &&  usergetmidnight($group->startdate) > usergetmidnight($deadline)) {
                $counter++;
                continue;
            }
            if ($group->enddate &&  usergetmidnight($group->enddate) < usergetmidnight($deadline)) {
                $counter++;
                continue;
            }
        }
        return $counter;
    }

    /**
     * @param $group
     * @param $users
     * @return int
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function count_enrollment_conflict($group, $users) {
        global $DB;

        $counter = 0;

        if (!$users) {
            return $counter;
        }

        if ($group->startdate == 0 && $group->enddate == 0) {
            return $counter;
        }

        list($insql, $params) = $DB->get_in_or_equal(array_keys($users));

        $sql = "SELECT ue.id, ue.userid, ue.timestart, ue.timeend 
                  FROM {user_enrolments} ue
                  JOIN {enrol} e
                    ON ue.enrolid = e.id
                 WHERE ue.status = 0
                   AND e.courseid = 695
                   AND ue.userid {$insql}";

        if (!$enrollments = $DB->get_records_sql($sql, $params)) {
            return $counter;
        }

        foreach ($enrollments as $enrollment) {
            if ($enrollment->timestart && $group->startdate > $enrollment->timestart && $group->enddate < $enrollment->timestart) {
                $counter++;
                continue;
            }
            if ($enrollment->timeend && $group->startdate > $enrollment->timeend && $group->enddate < $enrollment->timeend) {
                $counter++;
                continue;
            }
        }

        return $counter;
    }
}