<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Adhoc task for KICA report
 *
 * @package     report_ghs
 * @copyright   2020 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs\task;

use block_ned_teacher_tools\deadline_manager;
use report_ghs\helper;
use function block_ned_teacher_tools\get_grade_item;
use stdClass;
use core_tag_tag;

defined('MOODLE_INTERNAL') || die();

global $CFG;
require_once($CFG->dirroot . '/report/ghs/lib.php');
require_once($CFG->dirroot.'/local/kica/lib.php');
require_once($CFG->dirroot.'/grade/grading/lib.php');
require_once($CFG->dirroot.'/lib/gradelib.php');


/**
 * Class adhoc_ghs_missed_grade_report
 * @package report_ghs\task
 */
class adhoc_ghs_missed_grade_export extends base_ghs_adhoc_task {

    /**
     * @return bool
     */
    public function execute() {
        if (!static::can_execute()){
            return true;
        }

        $cd = $this->get_custom_data();
        self::export_report($cd->userid);

        return true;
    }

    /**
     * @return mixed
     * @throws \dml_exception
     */
    public static function get_task() {
        global $DB;

        $classname = self::class;
        if ($task = $DB->get_record('task_adhoc', ['classname' => '\\'.$classname])) {
            $task->customdata = json_decode($task->customdata);
        }

        return $task;
    }
    /**
     * @return string
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function export_report($userid = null, $client = null) {
        global $DB, $CFG;

        raise_memory_limit(MEMORY_EXTRA);

        if (!$client) {
            $client = helper::get_onedrive_client();
        }

        $columnsexport = array(
            'rowcount',
            'firstname',
            'lastname',
            'school',
            'userstatus',
            'excluded',
            'completed',
            'basecategory',
            'course',
            'groupname',
            'ctname',
            'otname',
            'duedate',
            'daysoverdue',
            'activityname',
            'activitygrade',
            'kicagrade',
            'skipped',
            'gcomp',
        );

        $params = null;
        $pageparams = [];

        $sql = "SELECT  r.*,
                u.firstname,
                u.lastname,
                coh.name  school,
                cor.shortname  course,
                cat.name  category,
                g.name  groupname,
                cat.parent basecategory,       
                if(r.ctid = -1, '*multiple', (SELECT u3.firstname FROM {user} u3 WHERE u3.id = r.ctid)) ctname,
                if(r.otid = -1, '*multiple', (SELECT u3.firstname FROM {user} u3 WHERE u3.id = r.otid)) otname,
                if(DATEDIFF(CURDATE(), FROM_UNIXTIME(r.duedate)) > 0, DATEDIFF(CURDATE(), FROM_UNIXTIME(r.duedate)), null) daysoverdue       
           FROM {report_ghs_missed_grade} r
     INNER JOIN {user} u ON r.userid = u.id
LEFT OUTER JOIN {cohort} coh ON r.chortid = coh.id
LEFT OUTER JOIN {course} cor ON r.courseid = cor.id
LEFT OUTER JOIN {course_categories} cat ON cor.category = cat.id
LEFT OUTER JOIN {groups} g ON r.groupid = g.id
          WHERE 0 = 0
            AND cat.parent = 81
            AND (r.graded/r.all*100) > 0 AND (r.graded/r.all*100) < 100 AND (r.activitygrade IS NULL OR r.kicagrade IS NULL)
            AND r.userstatus = 'Active'
            AND r.excluded = 0 
            AND r.completed = 0";

        // Delete firs rowcount column.
        array_shift($columnsexport);

        $headers = $columnsexport;

        foreach ($headers as $ckey => $column) {
            $headers[$ckey] = get_string($column, 'report_ghs');
        }

        $tempdir = $CFG->tempdir . '/phpexcel';
        if (!file_exists($tempdir)) {
            mkdir($tempdir, 0777, true);
        }

        $reportname = 'ghs_missed_grade';
        $filename = $tempdir . '/' . $reportname . '_' . date('Y-m-d') . '.csv';

        $outputcsv = fopen($filename, 'w+');

        // Output the column headings.
        fputcsv($outputcsv, $headers);

        $rs = $DB->get_recordset_sql($sql, $params);
        foreach ($rs as $tablerow) {
            $row = array();
            foreach ($columnsexport as $column) {
                $row[] = \report_ghs\helper::missed_grade_data($tablerow, $column, $counter, $pageparams, true);
            }
            fputcsv($outputcsv, $row);
        }
        $rs->close();

        if (!empty($outputcsv)) {
            fclose($outputcsv);
        }

        helper::upload_file_to_onedrive($client, $filename, ONEDRIVE_OLD_DATA);

        $newfile = $tempdir . '/' . $reportname . '.csv';
        if (copy($filename, $newfile)) {
            helper::upload_file_to_onedrive($client, $newfile, ONEDRIVE_NEW_DATA);
        }

        set_config('ghs_missed_grade_export', time(), 'report_ghs');

        if (empty($CFG->noemailever) && $userid) {
            $site = get_site();
            $from = \core_user::get_support_user();
            $from->firstname = $site->fullname;
            $from->lastname = "";

            $to = $DB->get_record('user', array('id' => $userid));
            $to->mailformat = 1;

            email_to_user($to, $from, get_string('ghsmissedgrade', 'report_ghs') . ' ' . get_string('export', 'report_ghs'), '', "TASK HAS BEEN EXECUTED!");
        }

        return true;
    }
}