<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Adhoc task for KICA report
 *
 * @package     report_ghs
 * @copyright   2020 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs\task;

use report_ghs\helper;
use report_ghs\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/report/ghs/lib.php');
require_once($CFG->dirroot.'/grade/grading/lib.php');
require_once($CFG->dirroot.'/lib/gradelib.php');


/**
 * Class adhoc_ghs_missed_grade_report
 * @package report_ghs\task
 */
class adhoc_ghs_missed_grade_report extends base_ghs_adhoc_task {

    /**
     * @return bool
     */
    public function execute() {
        if (!static::can_execute()){
            return true;
        }

        $cd = $this->get_custom_data();
        self::update_report($cd->userid ?? null);

        return true;
    }

    /**
     * @return mixed
     * @throws \dml_exception
     */
    public static function get_task() {
        global $DB;

        $classname = self::class;
        if ($task = $DB->get_record('task_adhoc', ['classname' => '\\'.$classname])) {
            $task->customdata = json_decode($task->customdata);
        }

        return $task;
    }

    /**
     * @param null $userid
     * @return bool
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function update_report($userid=null) {
        global $DB;

        $DB->execute("TRUNCATE TABLE {report_ghs_missed_grade}");

        $contextsystem = \context_system::instance();

        $otrole = $DB->get_record('role', array('shortname' => 'online-teacher'));
        $ctrole = $DB->get_record('role', array('shortname' => 'classroom-teacher'));
        $strsuspendedcourse = get_string('suspendedcourse', 'report_ghs');
        $strsuspendedsite = get_string('suspendedsite', 'report_ghs');
        $strcompleted = get_string('completed', 'report_ghs');
        $stractive = get_string('active', 'report_ghs');

        $courselist = helper::get_report_courses();

        list($insql, $params) = $DB->get_in_or_equal($courselist);

        // Courses.
        $sql = "SELECT c.* 
                  FROM {local_kica} k 
                  JOIN {course} c 
                    ON k.courseid = c.id
                 WHERE k.enabled = 1
                   AND c.id {$insql}
              ORDER BY c.shortname ASC";

        $sqlactivitygrade = "SELECT gg.id, gg.finalgrade, gg.rawgrademax, gg.excluded
                  FROM {grade_items} gi 
                  JOIN {grade_grades} gg 
                    ON gi.id = gg.itemid
                 WHERE gi.itemtype = 'mod' 
                   AND gi.itemmodule = ?
                   AND gi.iteminstance = ? 
                   AND gg.userid = ?";

        $sqlcohort = "SELECT c.* FROM {cohort} c JOIN {cohort_members} m ON c.id = m.cohortid WHERE c.contextid = ? AND m.userid = ?";

        $sqlgroup = "SELECT g.* FROM {groups} g JOIN {groups_members} m ON g.id = m.groupid WHERE g.courseid = ? AND m.userid = ?";

        $sqlcompletion = "SELECT cmc.* FROM {course_modules_completion} cmc INNER JOIN {course_modules} cm ON cmc.coursemoduleid = cm.id
                           WHERE cmc.coursemoduleid = ? AND cmc.userid = ? AND cm.deletioninprogress = 0";

        $rs = $DB->get_recordset_sql($sql, $params);
        $DM = NED::get_DM();

        foreach ($rs as $course) {
            $coursecontext = \context_course::instance($course->id);

            if (!$users = get_enrolled_users($coursecontext, 'mod/assign:submit', 0, 'u.*', 'u.id', 0, 0, false)) {
                continue;
            }

            $kica = NED::get_kica_enabled($course->id);
            if (!$kica){
                continue;
            }

            NED::ki_get_all_by_course($course->id, null, $users, true);
            NED::kg_get_grades_by_course($course->id, array_keys($users), false, true);
            foreach ($users as $index => $user) {
                $data = new \stdClass();
                $data->courseid = $course->id;
                $data->userid = $user->id;

                if ($cohort = $DB->get_record_sql($sqlcohort, [$contextsystem->id, $user->id], IGNORE_MULTIPLE)) {
                    $data->chortid = $cohort->id;
                }

                if ($groups = $DB->get_records_sql($sqlgroup, [$course->id, $user->id])) {
                    $data->groupid = (reset($groups))->id;
                }
                if ($ctusers = helper::get_role_users($ctrole->id, $coursecontext, ((isset($group)) ? $group->id : 0) )) {
                    $data->ctid = (count($ctusers) > 1) ? -1 : $ctusers[0]->id;
                }
                if ($otusers = helper::get_role_users($otrole->id, $coursecontext, ((isset($group)) ? $group->id : 0) )) {
                    $data->otid = (count($otusers) > 1) ? -1 : $otusers[0]->id;
                }

                // Fetching all modules in the course.
                $activities = NED::get_important_activities($course, $user->id);
                foreach ($activities as $mod) {
                    if (!$DM || !$DM::is_cm_enabled($mod->id)){
                        continue;
                    }

                    $module = $DM::get_dmm_by_cm($mod);
                    if (!$module){
                        continue;
                    }

                    $activityname = $module->get_activity_name();
                    // Eliminate activities do not start with number
                    if (preg_match('/^\d/', $activityname) !== 1) {
                        continue;
                    }

                    $deadline = $module->get_user_effective_access($user->id);
                    if (empty($deadline)) {
                        continue;
                    }

                    $data->duedate = $deadline;

                    $data->cmid = $mod->id;
                    $data->module = $mod->modname;
                    $data->instance = $mod->instance;
                    $data->activityname = $activityname;

                    if ($grade = $DB->get_record_sql($sqlactivitygrade, [$mod->modname, $mod->instance, $user->id])) {
                        $gradeval = grade_floatval($grade->finalgrade);
                        if (!is_null($gradeval)) {
                            $data->activitygrade = $gradeval;
                            $data->activitygrademax = $grade->rawgrademax;
                        }
                        $data->excluded = ($grade->excluded) ? 1 : 0;
                    }

                    if ($kicaitem = NED::ki_get_by_cm($mod, true)) {
                        if (!empty($kicaitem->id)){
                            $kica_grade = NED::kg_get_by_userid_itemid($user, $kicaitem);
                            $data->kicagrade = $kica_grade->get_finalgrade();
                            $data->kicagrademax = $kicaitem->get_grademax();
                        }
                    }

                    if ($tags = array_values(\core_tag_tag::get_item_tags_array('core', 'course_modules', $mod->id))) {
                        $_tags = [];
                        if (in_array('Summative', $tags)) {
                            $_tags[] = 'Summative';;
                        } else {
                            continue;
                        }
                        $data->activitytag = implode(', ', $_tags);
                    }

                    $data->userstatus = $stractive;
                    if ($user->suspended) {
                        $data->userstatus = $strsuspendedsite;
                    } else if (enrol_get_enrolment_end($course->id, $user->id) === false) {
                        $data->userstatus = $strsuspendedcourse;
                    }

                    $completion = $DB->get_record_sql($sqlcompletion, [$mod->id, $user->id]);
                    if (!empty($completion) && $completion->completionstate > 1) {
                        $data->completed = 1;
                    }

                    $data->timecreated = time();

                    $DB->insert_record('report_ghs_missed_grade', $data);
                }
            }

            NED::purge_course_depended_caches();
        }

        self::update_skipped_col();
        self::update_gcomp_col();

        $rs->close();

        set_config('kicalastupdate', time(), 'report_ghs');

        if (empty($CFG->noemailever) && $userid) {
            $site = get_site();
            $from = \core_user::get_support_user();
            $from->firstname = $site->fullname;
            $from->lastname = "";

            $to = $DB->get_record('user', array('id' => $userid));
            $to->mailformat = 1;

            email_to_user($to, $from, get_string('ghsmissedgrade', 'report_ghs'), '', "TASK HAS BEEN EXECUTED!");
        }

        return true;
    }

    /**
     * @param null $userid
     * @return bool
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function update_skipped_col() {
        global $CFG, $DB;
        $sql = "SELECT DISTINCT mg.userid, mg.courseid FROM {report_ghs_missed_grade} mg";

        $rs = $DB->get_recordset_sql($sql);
        foreach ($rs as $item) {

            $sql = "SELECT mg.id,
                           mg.userid,
                           mg.courseid,
                           mg.activitygrade,
                           mg.kicagrade,
                           mg.activityname
                      FROM {report_ghs_missed_grade} mg
                     WHERE mg.userid = ?
                       AND mg.courseid = ?";
            $records = $DB->get_records_sql($sql, [$item->userid, $item->courseid]);

            usort($records, function($a, $b) {
                return strnatcmp($a->activityname, $b->activityname);
            });

            $firstgraded = null;
            $lastgraded = null;
            foreach ($records as $record) {
                if (!is_null($record->activitygrade)) {
                    if (is_null($firstgraded)) {
                        $firstgraded = $record;
                    } else {
                        $lastgraded = $record;
                    }
                }
            }

            if (is_null($firstgraded) && is_null($lastgraded)) {
                $DB->execute('UPDATE {report_ghs_missed_grade} SET skipped = 0 WHERE userid = ? AND courseid = ?', [$item->userid, $item->courseid]);
            } else if (!is_null($firstgraded) && is_null($lastgraded)) {
                $skipped = 1;
                foreach ($records as $record) {
                    if ($record->id == $firstgraded->id) {
                        $skipped = 0;
                    }
                    self::set_skipped($record->id, $skipped);
                }
            } else if (!is_null($firstgraded) && !is_null($lastgraded)) {
                $first = 0;
                $last = 0;
                foreach ($records as $record) {
                    if ($record->id == $firstgraded->id) {
                        $first = 1;
                    }
                    if ($record->id == $lastgraded->id) {
                        $last = 1;
                    }
                    if (!$first) {
                        self::set_skipped($record->id, 1);
                    } else if ($first && !$last) {
                        if (!is_null($record->activitygrade)) {
                            self::set_skipped($record->id, 0);
                        } else {
                            self::set_skipped($record->id, 1);
                        }
                    } else {
                        self::set_skipped($record->id, 0);
                    }

                }
            }
        }
        $rs->close();

        return true;
    }
    /**
     * @param null $userid
     * @return bool
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function update_gcomp_col() {
        global $CFG, $DB;
        $sql = "SELECT DISTINCT mg.groupid, mg.cmid FROM {report_ghs_missed_grade} mg";
        $sqlgraded = "SELECT COUNT(1) FROM {report_ghs_missed_grade} mg WHERE mg.groupid = ? AND mg.cmid = ? AND (mg.activitygrade IS NOT NULL OR mg.kicagrade IS NOT NULL)";
        $sqlall = "SELECT COUNT(1) FROM {report_ghs_missed_grade} mg WHERE mg.groupid = ? AND mg.cmid = ?";

        $rs = $DB->get_recordset_sql($sql);
        foreach ($rs as $item) {
            $params = [$item->groupid, $item->cmid];
            $graded = $DB->count_records_sql($sqlgraded, $params);
            $all = $DB->count_records_sql($sqlall, $params);
            $gcompstr = "$graded / $all";
            $DB->execute('UPDATE {report_ghs_missed_grade} SET gcomp = ?, graded = ?, `all` = ? WHERE groupid = ? AND cmid = ?', [$gcompstr, $graded, $all, $item->groupid, $item->cmid]);
        }
        $rs->close();

        return true;
    }

    public static function set_skipped($id, $value) {
        global $DB;
        $rec = new \stdClass();
        $rec->id = $id;
        $rec->skipped = $value;
        $DB->update_record('report_ghs_missed_grade', $rec);
    }
}