<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Adhoc task for KICA report
 *
 * @package     report_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs\task;

use report_ghs\helper;
use report_ghs\onedrivetoken;
use stdClass;
use Krizalys\Onedrive\Onedrive;

defined('MOODLE_INTERNAL') || die();

global $CFG;
require_once($CFG->dirroot . '/report/ghs/lib.php');
require_once($CFG->dirroot.'/local/kica/lib.php');
require_once($CFG->dirroot.'/grade/grading/lib.php');
require_once($CFG->dirroot.'/lib/gradelib.php');


/**
 * Class adhoc_ghs_grades_report
 * @package report_ghs\task
 */
class adhoc_ghs_submissions_export extends base_ghs_adhoc_task {

    /**
     * @return bool
     */
    public function execute() {
        if (!static::can_execute()){
            return true;
        }

        $cd = $this->get_custom_data();
        self::export_report($cd->userid);

        return true;
    }

    /**
     * @return mixed
     * @throws \dml_exception
     */
    public static function get_task() {
        global $DB;

        $classname = self::class;
        if ($task = $DB->get_record('task_adhoc', ['classname' => '\\'.$classname])) {
            $task->customdata = json_decode($task->customdata);
        }

        return $task;
    }


    /**
     * @return string
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function export_report($userid = null, $client = null) {
        global $DB, $CFG;

        raise_memory_limit(MEMORY_EXTRA);

        if (!$client) {
            $client = helper::get_onedrive_client();
        }

        foreach (['Submission_records_course_school', 'Submission_records_activity', 'Submission_records_student'] as $reportname) {

            switch ($reportname) {
                case 'Submission_records_course_school':
                    $columnsexport = array(
                        'rowcount',
                        'course',
                        'fcategory',
                        'lcategory',
                        'school',
                        'groupname',
                        'wrongsubmissions',
                        'latesubmissions',
                        'misseddeadlines',
                    );

                    $sql = "SELECT cor.shortname AS course,
                                   cat.name category,
                                   cat.path categorypath,
                                   coh.name AS school,
                                   g.name  groupname,
                                   SUM(r.wrongsubmission) wrongsubmissions,
                                   SUM(r.latesubmission) latesubmissions,
                                   SUM(r.misseddeadline) misseddeadlines 
                              FROM {report_ghs_submissions} r
                         LEFT JOIN {cohort} coh ON r.cohortid = coh.id
                         LEFT JOIN {course} cor ON r.courseid = cor.id
                         LEFT JOIN {course_categories} cat ON cor.category = cat.id 
                         LEFT JOIN {groups} g ON r.groupid = g.id
                          GROUP BY r.courseid, r.cohortid, r.groupid";
                    break;
                case 'Submission_records_activity':
                    $columnsexport = array(
                        'rowcount',
                        'course',
                        'fcategory',
                        'lcategory',
                        'school',
                        'activityname',
                        'wrongsubmissions',
                        'latesubmissions',
                        'misseddeadlines',
                    );

                    $sql = "SELECT cor.shortname AS course,
                                   cat.name category,
                                   cat.path categorypath,
                                   coh.name AS school,
                                   r.activityname,
                                   SUM(r.wrongsubmission) wrongsubmissions,
                                   SUM(r.latesubmission) latesubmissions,
                                   SUM(r.misseddeadline) misseddeadlines 
                              FROM {report_ghs_submissions} r
                         LEFT JOIN {cohort} coh ON r.cohortid = coh.id
                         LEFT JOIN {course} cor ON r.courseid = cor.id
                         LEFT JOIN {course_categories} cat ON cor.category = cat.id 
                         LEFT JOIN {groups} g ON r.groupid = g.id
                          GROUP BY r.courseid, r.cohortid, r.cmid, r.activityname";
                    break;
                case 'Submission_records_student':
                    $columnsexport = array(
                        'rowcount',
                        'firstname',
                        'lastname',
                        'school',
                        'wrongsubmissions',
                        'latesubmissions',
                        'misseddeadlines',
                    );

                    $sql = "SELECT coh.name AS school,
                                   u.firstname,
                                   u.lastname,
                                   SUM(r.wrongsubmission) wrongsubmissions,
                                   SUM(r.latesubmission) latesubmissions,
                                   SUM(r.misseddeadline) misseddeadlines 
                              FROM {report_ghs_submissions} r
                        INNER JOIN {user} u ON r.userid = u.id
                         LEFT JOIN {cohort} coh ON r.cohortid = coh.id
                           GROUP BY r.cohortid, r.userid";
                    break;

                default: continue 2;
            }


            $params = null;
            $pageparams = [];

            // Delete firs rowcount column.
            array_shift($columnsexport);

            $headers = $columnsexport;

            foreach ($headers as $ckey => $column) {
                $headers[$ckey] = get_string($column, 'report_ghs');
            }

            $tempdir = $CFG->tempdir . '/phpexcel';
            if (!file_exists($tempdir)) {
                mkdir($tempdir, 0777, true);
            }

            $filename = $tempdir . '/' . $reportname . '_' . date('Y-m-d') . '.csv';

            $outputcsv = fopen($filename, 'w+');

            // Output the column headings.
            fputcsv($outputcsv, $headers);

            $rs = $DB->get_recordset_sql($sql, $params);
            foreach ($rs as $tablerow) {
                $row = array();
                foreach ($columnsexport as $column) {
                    $row[] = \report_ghs\helper::kica_data($tablerow, $column, $counter, $pageparams, true);
                }
                fputcsv($outputcsv, $row);
            }
            $rs->close();

            if (!empty($outputcsv)) {
                fclose($outputcsv);
            }

            helper::upload_file_to_onedrive($client, $filename, ONEDRIVE_OLD_DATA);

            $newfile = $tempdir . '/' . $reportname . '.csv';
            if (copy($filename, $newfile)) {
                helper::upload_file_to_onedrive($client, $newfile, ONEDRIVE_NEW_DATA);
            }
        }

        set_config('ghs_submissions_export', time(), 'report_ghs');

        if (empty($CFG->noemailever) && $userid) {
            $site = get_site();
            $from = \core_user::get_support_user();
            $from->firstname = $site->fullname;
            $from->lastname = "";

            $to = $DB->get_record('user', array('id' => $userid));
            $to->mailformat = 1;

            email_to_user($to, $from, get_string('ghsgrades', 'report_ghs') .' '. get_string('export', 'report_ghs'), '', "TASK HAS BEEN EXECUTED!");
        }

        return true;
    }
}