<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Adhoc task for KICA report
 *
 * @package     report_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs\task;

use local_ned_controller\ned_grade_controller as NGC;
use local_schoolmanager\output\school;
use local_schoolmanager\school_handler as SH;
use local_schoolmanager\school_manager as SM;
use report_ghs\helper;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/report/ghs/lib.php');
require_once($CFG->dirroot.'/local/kica/lib.php');
require_once($CFG->dirroot.'/grade/grading/lib.php');
require_once($CFG->dirroot.'/lib/gradelib.php');

/**
 * Class adhoc_ghs_grades_report
 * @package report_ghs\task
 */
class adhoc_ghs_submissions_report extends base_ghs_adhoc_task {

    /**
     * @return bool
     */
    public function execute() {
        if (!static::can_execute()){
            return true;
        }

        $cd = $this->get_custom_data();
        self::update_report($cd->userid ?? null);
        return true;
    }

    /**
     * @return mixed
     * @throws \dml_exception
     */
    public static function get_task() {
        global $DB;

        $classname = self::class;
        if ($task = $DB->get_record('task_adhoc', ['classname' => '\\'.$classname])) {
            $task->customdata = json_decode($task->customdata);
        }

        return $task;
    }

    /**
     * @param null $userid
     * @return bool
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function update_report($userid=null) {
        global $CFG, $DB, $USER;

        $cfg = get_config('report_ghs');

        $USER = get_admin();

        $DB->execute("TRUNCATE TABLE {report_ghs_submissions}");

        $sqlgroup = "SELECT g.* FROM {groups} g JOIN {groups_members} m ON g.id = m.groupid WHERE g.courseid = ? AND m.userid = ?";

        $sh = new SH();
        $schools = $sh->get_schools();

        foreach (NGC::STATUSES as $status => $st_lang_key){
            if (NGC::HIDDEN_STATUSES[$status] ?? false) continue;
            $show_statuses[] = $status;
        }

        foreach ($schools as $school) {
            $SM = new SM();
            if (!$users = $SM->get_school_students($school->id, true)) {
                continue;
            }

            $data = new \stdClass();
            $data->cohortid = $school->id;

            foreach ($users as $user) {
                $data->userid = $user->id;

                if (!$grades = $DB->get_records('local_ned_controller_grades', ['userid' => $user->id])) {
                    continue;
                }

                foreach ($grades as $grade) {
                    $data->ncgid = $grade->id;
                    $data->courseid = $grade->courseid;
                    $data->cmid = $grade->cmid;

                    if ($cm = get_coursemodule_from_id('', $grade->cmid, $grade->courseid)) {
                        $data->module = $cm->modname;
                        $data->instance = $cm->instance;
                        $data->activityname = $cm->name;
                    }

                    if ($groups = $DB->get_records_sql($sqlgroup, [$grade->courseid, $user->id])) {
                        $data->groupid = (reset($groups))->id;
                    }

                    if (!empty($show_statuses) && in_array($grade->status, $show_statuses)) {
                        $data->wrongsubmission = ($grade->reason == NGC::REASON_FILE) ? 1 : 0;
                        $data->latesubmission = ($grade->reason == NGC::REASON_SUBMISSION && $grade->grade_type == NGC::GT_DEDUCTION) ? 1 : 0;
                        $data->misseddeadline = ($grade->reason == NGC::REASON_SUBMISSION && $grade->grade_type == NGC::GT_AWARD_ZERO) ? 1 : 0;
                    }

                    $data->timecreated = time();

                    $DB->insert_record('report_ghs_submissions', $data);
                }
            }
        }

        set_config('ghssubmissionsupdate', time(), 'report_ghs');

        if (empty($CFG->noemailever) && $userid) {
            $site = get_site();
            $from = \core_user::get_support_user();
            $from->firstname = $site->fullname;
            $from->lastname = "";

            $to = $DB->get_record('user', array('id' => $userid));
            $to->mailformat = 1;

            email_to_user($to, $from, get_string('ghssubmissions', 'report_ghs'), '', "TASK HAS BEEN EXECUTED!");
        }

        return true;
    }
}