<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Scheduled task
 *
 * @package     report_ghs
 * @copyright   2023 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs\task;

use report_ghs\helper;
use local_schoolmanager\school;
use report_ghs\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/**
 * Class update_reports
 * @package report_ghs\task
 */
class sync_user_info_data extends \core\task\scheduled_task {

    /**
     * @return string
     * @throws \coding_exception
     */
    public function get_name() {
        return get_string('syncuserinfodata', 'report_ghs');
    }

    /**
     * @return bool
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public function execute() {
        global $DB, $USER;

        $USER = get_admin();

        $rolefield = NED::ENGLISH_PROFICIENCY_DEFAUL_ROLE_FIELD;
        $rolename = NED::ENGLISH_PROFICIENCY_DEFAUL_ROLE;

        $sql = "SELECT DISTINCT d.userid FROM {user_info_data} d";
        $rs = $DB->get_recordset_sql($sql);
        foreach ($rs as $userid => $ignore) {
            $data = $DB->get_records('user_info_data', ['userid' => $userid]);

            $user = \core_user::get_user($userid);

            $rec = new \stdClass();
            $rec->userid = $userid;
            $rec->firstname = $user->firstname ?? null;
            $rec->lastname = $user->lastname ?? null;

            $skip = true;

            foreach ($data as $datum) {
                if (isset(NED::ENGLISH_PROFICIENCY_FIELDS[$datum->fieldid])) {
                    $field = NED::ENGLISH_PROFICIENCY_FIELDS[$datum->fieldid];
                    $rec->{$field} = $datum->data;
                    $skip = false;
                }
            }

            if (!$skip) {
                if ($proficiency = $DB->get_record('report_ghs_english_proficien', ['userid' => $userid])) {
                    $rec->id = $proficiency->id;
                    $DB->update_record('report_ghs_english_proficien', $rec);
                } else {
                    $DB->insert_record('report_ghs_english_proficien', $rec);
                }
            }
        }
        $rs->close();

        // Score updates
        $sqlcourse = "SELECT e.courseid 
                        FROM {user_enrolments} ue
                        JOIN {enrol} e
                          ON ue.enrolid = e.id
                        JOIN {course} c
                          ON e.courseid = c.id
                       WHERE userid = :userid AND c.shortname LIKE :shortname";
        
        $rs = $DB->get_recordset('report_ghs_english_proficien', [$rolefield => $rolename]);
        foreach ($rs as $proficiency) {
            if ($schoolid = NED::get_user_school($proficiency->userid, true)) {
                if ($school = new school($schoolid)) {
                    $DB->set_field('report_ghs_english_proficien', 'school', $school->get('code') ?? null, ['id' => $proficiency->id]);
                }
            }

            foreach (NED::ENGLISH_PROFICIENCY_COURSES as $coursecode) {
                if ($courses = helper::get_proficiency_courses($proficiency->userid, $coursecode)) {
                    $course = reset($courses);
                    $kica_average_data = NED::kg_get_kica_average($course->courseid, $proficiency->userid, null, null, null);
                    $kicagrades = NED::kg_get_course_average($course->courseid, $proficiency->userid, null, false, $kica_average_data);
                    $kicaavg = $kicagrades[NED::FINALGRADE] ?? null;

                    $DB->set_field('report_ghs_english_proficien', $coursecode.'_grade', $kicaavg, ['id' => $proficiency->id]);
                }
            }
        }
        $rs->close();

        return true;
    }
}