<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * RADIX
 *
 * @package     report_ghs
 * @copyright   2024 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once(dirname(dirname(dirname(__FILE__))) . '/config.php');
require_once("$CFG->libdir/phpspreadsheet/vendor/autoload.php");
require_once($CFG->libdir.'/adminlib.php');

use local_schoolmanager\school_handler as SH;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use report_ghs\helper;
use local_schoolmanager\output\reports_header;
use report_ghs\shared_lib as NED;

// Paging options.
$page      = optional_param('page', 0, PARAM_INT);
$perpage   = optional_param('perpage', 20, PARAM_INT);
$sort      = optional_param('sort', 'course', PARAM_ALPHANUM);
$dir       = optional_param('dir', 'ASC', PARAM_ALPHA);
// Action.
$action    = optional_param('action', false, PARAM_ALPHA);
$search    = optional_param('search', '', PARAM_TEXT);

// Filters.
$schoolid  = optional_param('schoolid', 0, PARAM_INT);
$filtercourse = optional_param('filtercourse', 0, PARAM_INT);
$filterclass = optional_param('filterclass', 0, PARAM_INT);
$filterstudent = optional_param('filterstudent', 0, PARAM_INT);
$filtertype = optional_param('filtertype', 0, PARAM_INT);
$filterdeadlinetype = optional_param('filterdeadlinetype', 0, PARAM_INT);
$filterperiod = optional_param('filterperiod', 1, PARAM_INT);

$cfg = get_config('report_ghs');

require_login(null, false);
$contextsystem = context_system::instance();

// Permission.
if (!has_any_capability(['report/ghs:viewclassdeadlinesallschools', 'report/ghs:viewclassdeadlinesownschool'], $contextsystem)) {
    throw new required_capability_exception($contextsystem, 'report/ghs:viewclassdeadlinesallschools', 'nopermissions', '');
}

$thispageurl = new moodle_url('/report/ghs/ghs_class_deadlines.php');
$fullpageurl = new moodle_url('/report/ghs/ghs_class_deadlines.php', [
    'page' => $page,
    'perpage' => $perpage,
    'sort' => $sort,
    'dir' => $dir,
    'action' => $action,
    'schoolid' => $schoolid,
    'filtercourse' => $filtercourse,
    'filterclass' => $filterclass,
    'filterstudent' => $filterstudent,
    'filtertype' => $filtertype,
    'filterdeadlinetype' => $filterdeadlinetype,
    'filterperiod' => $filterperiod,
]);
$report = basename($thispageurl->out(), '.php');

$PAGE->set_url($thispageurl);
$PAGE->set_pagelayout('schoolmanager');
$PAGE->set_context($contextsystem);
$PAGE->add_body_classes(['path-report-ghs', 'path-blocks-ned_teacher_tools']);

$name = get_string('ghsclassdeadlines', 'report_ghs');
$title = get_string('ghsclassdeadlines', 'report_ghs');
$heading = $SITE->fullname;

// Breadcrumb.
if (has_capability('report/ghs:viewclassdeadlinesallschools', $contextsystem)) {
    admin_externalpage_setup('reportghsclassdeadlines', null, null, '', ['pagelayout' => 'schoolmanager']);
}

$PAGE->set_title($title);
$PAGE->set_heading($heading);

$datacolumns = array(
    'id' => 'r.id',
    'school' => 'coh.name',
    'course' => 'c.shortname',
    'courseid' => 'c.id',
    'class' => 'g.name',
    'activity' => "(CASE
                    WHEN r.module = 'assign' THEN (SELECT q.name FROM {assign} q WHERE q.id = r.instance)
                    WHEN r.module = 'quiz' THEN (SELECT q.name FROM {quiz} q WHERE q.id = r.instance)    
                    ELSE ''
                END)",
    'deadline' => 'r.deadline',
    'deadlinetype' => 'r.deadlinetype',
    'countdown' => '(r.deadline - UNIX_TIMESTAMP())',
    'schoolid' => 'r.cohortid',
    'groupid' => 'r.groupid',
    'userid' => 'r.userid',
    'module' => 'r.module',
);

$params = [];

// Filter.
$where = '';

if (!has_capability('report/ghs:viewclassdeadlinesallschools', $contextsystem)) {
    $sh = new SH();
    if (!$schools = $sh->get_schools()) {
        $where .= " AND 0=1";
    } else {
        list($insql, $aparams) = $DB->get_in_or_equal(array_keys($schools), SQL_PARAMS_NAMED);
        $where .= " AND " . $datacolumns['schoolid'] . " {$insql}";
        $params = array_merge($params, $aparams);
    }
} else if ($schoolid) {

    $where .= " AND ".$datacolumns['schoolid']." = :schoolid";
    $params['schoolid'] = $schoolid ?? 0;
}

if ($filtercourse) {
    $where .= " AND ".$datacolumns['courseid']." = :course";
    $params['course'] = $filtercourse;
}

if ($filterclass) {
    $where .= " AND ".$datacolumns['groupid']." = :class";
    $params['class'] = $filterclass;
}

if ($filterstudent) {
    $where .= " AND ".$datacolumns['userid']." = :student";
    $params['student'] = $filterstudent;
}

if ($filtertype) {
    $where .= " AND ".$datacolumns['module']." = :module";
    if ($filtertype == 1) {
        $params['module'] = 'assign';
    } else {
        $params['module'] = 'quiz';
    }
}

if ($filterdeadlinetype) {
    $where .= " AND ".$datacolumns['deadlinetype']." = :deadlinetype";
    if ($filterdeadlinetype == 1) {
        $params['deadlinetype'] = 'class';
    } else {
        $params['deadlinetype'] = 'extension';
    }
}

if ($filterperiod) {
    if ($filterperiod == 1) { // nextsixtydays
        $where .= " AND ".$datacolumns['deadline']." > :deadline1";
        $where .= " AND ".$datacolumns['deadline']." < :deadline2";
        $params['deadline1'] = time();
        $params['deadline2'] = time() + 60 * DAYSECS;
    } else { // lastsixtydays
        $where .= " AND ".$datacolumns['deadline']." > :deadline1";
        $where .= " AND ".$datacolumns['deadline']." < :deadline2";
        $params['deadline1'] = time() - 60 * DAYSECS;
        $params['deadline2'] = time();
    }
}

// Sort.
$order = '';
if ($sort) {
    $order = " ORDER BY $datacolumns[$sort] $dir";
}

$pageparams = array();

// Count records for paging.
$countsql = "SELECT COUNT(1) 
           FROM (SELECT d.*, COUNT(1) AS numberofstudents FROM {report_ghs_class_deadlines} d GROUP BY d.cmid, d.groupid, d.deadlinetype, d.deadline) r
     INNER JOIN {course} c ON r.courseid = c.id
LEFT OUTER JOIN {cohort} coh ON r.cohortid = coh.id
LEFT OUTER JOIN {groups} g ON r.groupid = g.id
          WHERE 0=0                 
            $where";

$totalcount = $DB->count_records_sql($countsql, $params);

// Table columns.
$columns = array(
    'school',
    'course',
    'class',
    'activity',
    'deadline',
    'students',
    'deadlinetype',
    'countdown'
);

$sql = " SELECT r.*,
                coh.idnumber school,
                g.name groupname,
                c.shortname course,
                c.id courseid,
                (r.deadline -  UNIX_TIMESTAMP()) countdown,
                CASE
                    WHEN r.module = 'assign' THEN (SELECT q.name FROM {assign} q WHERE q.id = r.instance)
                    WHEN r.module = 'quiz' THEN (SELECT q.name FROM {quiz} q WHERE q.id = r.instance)    
                    ELSE ''
                END as 'activity'
           FROM (SELECT d.*, COUNT(1) AS numberofstudents FROM {report_ghs_class_deadlines} d GROUP BY d.cmid, d.groupid, d.deadlinetype, d.deadline) r
     INNER JOIN {course} c ON r.courseid = c.id
LEFT OUTER JOIN {cohort} coh ON r.cohortid = coh.id
LEFT OUTER JOIN {groups} g ON r.groupid = g.id
          WHERE 0=0
                $where              
                $order";

if ($action == 'excel') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);

    $table = new stdClass();
    $table->head = $columns;

    $counter = 0;
    $filename = $report.'_'.(date('Y-m-d'));
    $downloadfilename = clean_filename($filename);

    $workbook = new Spreadsheet();
    $myxls = $workbook->setActiveSheetIndex(0);

    $numberofcolumns = count($table->head);

    // Header row.
    foreach ($table->head as $key => $heading) {
        $cell = Coordinate::stringFromColumnIndex($key + 1) . '1'; // A1 cell address.
        $myxls->setCellValue($cell, str_replace("\n", ' ', htmlspecialchars_decode(strip_tags(nl2br(get_string($heading, 'report_ghs'))))));
    }

    // Data rows.
    $rs = $DB->get_recordset_sql($sql, $params);
    $rownum = 2;
    foreach ($rs as $tablerow) {
        $row = array();
        $columnum = 1;
        foreach ($table->head as $column) {
            $data = \report_ghs\helper::class_deadlines_data($tablerow, $column, $counter, $pageparams, true);
            $cell = Coordinate::stringFromColumnIndex($columnum) . $rownum; // A2 cell address.
            if (preg_match("/^[fh]tt?p:\/\//", $data)) {
                $myxls->setCellValue($cell, 'Link');
                $myxls->getCell($cell)->getHyperlink()->setUrl($data);
            } else {
                $myxls->setCellValue($cell, $data);
            }
            $columnum++;
        }
        $rownum++;
    }
    $rs->close();


    // Freeze header.
    $myxls->freezePane('A2');

    // Filter.
    $myxls->setAutoFilter(
        $myxls->calculateWorksheetDimension()
    );

    // Auto column width calculation.
    foreach (range('A', $myxls->getHighestDataColumn()) as $col) {
        $myxls->getColumnDimension($col)->setAutoSize(true);
    }

    // Header format.
    $styleArray = array(
        'font' => array(
            'bold' => true,
        ),
        'fill' => array(
            'fillType' => Fill::FILL_SOLID,
            'color' => array(
                'argb' => 'FFFFF000',
            ),
        ),
    );
    $myxls->getStyle('A1:'.Coordinate::stringFromColumnIndex($numberofcolumns).'1')->applyFromArray($styleArray);

    // Rename worksheet
    $myxls->setTitle('export');

    // Set active sheet index to the first sheet, so Excel opens this as the first sheet
    $workbook->setActiveSheetIndex(0);

    $objWriter = new Xlsx($workbook);

    if (ob_get_length()) {
        ob_end_clean();
    }
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment;filename="'.$downloadfilename.'.xlsx"');
    header('Cache-Control: max-age=0');

    ob_end_clean();
    $objWriter->save('php://output');
    exit;
} else if ($action == 'csv') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);
    $table = new stdClass();

    $headers = $columns;

    foreach ($headers as $ckey => $column) {
        $headers[$ckey] = get_string($column, 'report_ghs');
    }

    if (ob_get_length()) {
        ob_end_clean();
    }
    // Output headers so that the file is downloaded rather than displayed.
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=ghs_enrollment_'.(date('Y-m-d')).'.csv');

    // Create a file pointer connected to the output stream.
    $outputcsv = fopen('php://output', 'w');

    // Output the column headings.
    fputcsv($outputcsv, $headers);

    $counter = 0;

    $rs = $DB->get_recordset_sql($sql, $params);
    foreach ($rs as $tablerow) {
        $row = array();
        foreach ($columns as $column) {
            $row[] = \report_ghs\helper::class_deadlines_data($tablerow, $column, $counter, $pageparams, true);
        }
        fputcsv($outputcsv, $row);
    }
    $rs->close();
    exit;
} else {

    foreach ($columns as $column) {
        $string[$column] = get_string($column, 'report_ghs');
        if ($sort != $column) {
            $columnicon = "";
            $columndir = "ASC";
        } else {
            $columndir = $dir == "ASC" ? "DESC" : "ASC";
            $columnicon = ($dir == "ASC") ? "sort_asc" : "sort_desc";
            $columnicon = $OUTPUT->pix_icon('t/'.$columnicon, '', 'moodle', array('class' => 'iconsort'));
        }
        if (($column == 'rowcount') || ($column == 'action') || ($column == 'gradebooklink') || ($column == 'kicagradebooklink')) {
            $$column = $string[$column];
        } else {
            $sorturl = clone $fullpageurl;
            $sorturl->param('perpage', $perpage);
            $sorturl->param('sort', $column);
            $sorturl->param('dir', $columndir);
            $sorturl->param('search', $search);

            $$column = html_writer::link($sorturl->out(false), $string[$column]).' '.$columnicon;
        }
    }

    $table = new html_table();

    $table->head = array();
    $table->wrap = array();
    $table->attributes = ['class' => 'nedtable fullwidth'];

    foreach ($columns as $column) {
        $table->head[$column] = $$column;
        $table->wrap[$column] = '';
    }

    // Override cell wrap.
    $table->wrap['action'] = 'nowrap';

    $tablerows = $DB->get_records_sql($sql, $params, $page * $perpage, $perpage);

    $counter = ($page * $perpage);

    foreach ($tablerows as $tablerow) {
        $row = new html_table_row();

        foreach ($columns as $column) {
            $varname = 'cell'.$column;
            $$varname = new html_table_cell(\report_ghs\helper::class_deadlines_data($tablerow, $column, $counter, $pageparams));
            if ($tablerow->deadlinetype == NED::DEADLINE_TYPE_EXTENSION && (in_array($column , ['students', 'deadlinetype']))) {
                $$varname->attributes = ['class' => 'bg-red'];
            }
        }

        $row->cells = array();
        foreach ($columns as $column) {
            $varname = 'cell' . $column;
            $row->cells[$column] = $$varname;
        }
        $table->data[] = $row;

    }

    echo $OUTPUT->header();
    echo html_writer::start_div('page-content-wrapper');
    echo $OUTPUT->render(new reports_header());
    echo html_writer::tag('h1', $title, array('class' => 'page-title'));

    $reportcls = '\report_ghs\task\adhoc_'.$report.'_report';

    if (class_exists($reportcls)) {
        if ($task = $reportcls::get_task()) {
            echo html_writer::tag('div', get_string('taskwarning', 'report_ghs', userdate($task->customdata->timecreated)), array('class' => 'alert alert-info'));
        } else {
            $a = new stdClass();
            $a->lastupdate = userdate($cfg->classdeadlineslastupdate);
            $a->updateurl = (new moodle_url('/report/ghs/update_report.php', ['report' => $report]))->out();
            echo html_writer::tag('div', get_string('lastupdatewarning', 'report_ghs', $a), array('class' => 'alert alert-info'));
        }
    }

    \report_ghs\helper::print_export_task_info($report, $cfg);

    $handler = new SH();
    $schoolfilter = $handler->get_control_form($schoolid, $fullpageurl, false, true);

    // Filter form.
    $searchformurl = new moodle_url('/report/ghs/ghs_class_deadlines.php');


    // Courses
    $sql = "SELECT DISTINCT r.courseid, c.shortname name
              FROM {report_ghs_class_deadlines} r
              JOIN {course} c ON r.courseid = c.id
          ORDER BY c.shortname ASC";
    $courseoptions = ['0' => get_string('all')] + $DB->get_records_sql_menu($sql, $params);

    // Classes
    $sql = "SELECT DISTINCT r.groupid, g.name
              FROM {report_ghs_class_deadlines} r
              JOIN {groups} g ON r.groupid = g.id
          ORDER BY g.name ASC";
    $classoptions =  ['0' => get_string('all')] + $DB->get_records_sql_menu($sql, $params);


    // Students
    $sql = "SELECT DISTINCT r.userid, CONCAT(u.firstname, ' ',  u.lastname) name
              FROM {report_ghs_class_deadlines} r
              JOIN {user} u ON r.userid = u.id
          ORDER BY u.lastname ASC";
    $studentoptions = ['0' => get_string('all')] + $DB->get_records_sql_menu($sql, $params);

    $typeoptions = [
        0 => get_string('all', 'report_ghs'),
        1 => get_string('assign', 'report_ghs'),
        2 => get_string('test', 'report_ghs'),
    ];
    
    $deadlinetypeoptions = [
        0 => get_string('all', 'report_ghs'),
        1 => get_string('class', 'report_ghs'),
        2 => get_string('extension', 'report_ghs'),
    ];

    $filterperiodoptions = [
        0 => get_string('all', 'report_ghs'),
        1 => get_string('nextsixtydays', 'report_ghs'),
        2 => get_string('lastsixtydays', 'report_ghs'),
    ];

    $searchform = html_writer::tag('form',
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'sesskey',
            'value' => sesskey(),
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'perpage',
            'value' => $perpage,
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'sort',
            'value' => $sort,
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'dir',
            'value' => $dir,
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'schoolid',
            'value' => $schoolid,
        )).

        html_writer::start_div('form-inline').
        html_writer::start_div('form-group').

        html_writer::start_div('p-2').
        $schoolfilter.
        html_writer::end_div().

        // Course.
        NED::single_autocomplete($fullpageurl, 'filtercourse', $courseoptions, $filtercourse, get_string('course', 'report_ghs'), 'mb-2 mr-sm-2').

        // Class.
        NED::single_autocomplete($fullpageurl, 'filterclass', $classoptions, $filterclass, get_string('class', 'report_ghs'), ['class' => 'mb-2 mr-sm-2']).

        // Student.
        NED::single_autocomplete($fullpageurl, 'filterstudent', $studentoptions, $filterstudent, get_string('student', 'report_ghs'), 'mb-2 mr-sm-2').

        // Activity type.
        NED::single_autocomplete($fullpageurl, 'filtertype', $typeoptions, $filtertype, get_string('activitytype', 'report_ghs'), ['class' => 'mb-2 mr-sm-2']).
        html_writer::end_div().
        html_writer::start_div('form-group').
        // Deadline type.
        NED::single_autocomplete($fullpageurl, 'filterdeadlinetype', $deadlinetypeoptions, $filterdeadlinetype, get_string('deadlinetype', 'report_ghs'), ['class' => 'mb-2 mr-sm-2']).

        // Period.
        NED::single_autocomplete($fullpageurl, 'filterperiod', $filterperiodoptions, $filterperiod, get_string('period', 'report_ghs'), ['class' => 'mb-2 mr-sm-2']).


        html_writer::empty_tag('input', array(
            'type' => 'submit',
            'value' => get_string('filter', 'report_ghs'),
            'class' => 'btn btn-primary form-control mb-2 mr-sm-2',
        )).

        html_writer::link($searchformurl,
            get_string('reset', 'report_ghs'), array('class' => 'btn btn-secondary form-control mb-2 mr-sm-2 filter-reset')
        ).
        html_writer::end_div().
        html_writer::end_div(),
        array(
            'action' => $searchformurl->out(),
            'method' => 'post',
            'class' => '_form-inline mb-2 mt-2',
            'autocomplete' => 'off'
        )
    );

    echo html_writer::div($searchform, 'search-form-wrapper', array('id' => 'search-form'));

    $pagingurl = clone $fullpageurl;
    $pagingurl->remove_params('page');
    $pagingurl->remove_params('action');

    $pagingbar = new paging_bar($totalcount, $page, $perpage, $pagingurl, 'page');

    // Export link.
    $exporexcelurl = $thispageurl;
    $exporexcelurl->remove_all_params();
    $exporexcelurl->param('schoolid', $schoolid);
    $exporexcelurl->param('action', 'excel');
    $exportexceliconurl = $OUTPUT->image_url('f/spreadsheet');
    $exportexcelicon = html_writer::img($exportexceliconurl, '', array('width' => '16', 'height' => '16'));
    $exportexceliconlink = html_writer::link($exporexcelurl, $exportexcelicon);
    $exportexcellink = html_writer::link($exporexcelurl, 'XLS');

    $exportexturl = $exporexcelurl;
    $exportexturl->param('action', 'csv');
    $exporttexticonurl = $OUTPUT->image_url('f/text');
    $exporttexticon = html_writer::img($exporttexticonurl, '', array('width' => '16', 'height' => '16'));
    $exporttexticonlink = html_writer::link($exportexturl, $exporttexticon);
    $exporttextlink = html_writer::link($exportexturl, 'CSV');

    $exportbuttons =  html_writer::div(
        get_string('export', 'report_ghs').' :&nbsp;&nbsp;'.
        $exportexceliconlink.'&nbsp;'.$exportexcellink.'&nbsp;|&nbsp; '
        .$exporttexticonlink.'&nbsp;'.$exporttextlink, 'export-link-wrapper',
        array('style' => 'text-align:center;')
    );

    if ($outputpagingbar = $OUTPUT->render($pagingbar)) {
        echo $outputpagingbar;
    } else {
        echo html_writer::tag('div', '', ['class' => 'dummy-pagination']);
    }
    echo html_writer::div($exportbuttons, 'export-link-wrapper-top');
    echo html_writer::table($table);
    echo $outputpagingbar;
    echo $exportbuttons;

    echo html_writer::end_div(); // Main wrapper.
    echo $OUTPUT->footer();
}