<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * RADIX
 *
 * @package     report_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once(dirname(dirname(dirname(__FILE__))) . '/config.php');
require_once("$CFG->libdir/phpspreadsheet/vendor/autoload.php");
require_once($CFG->libdir.'/adminlib.php');

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use report_ghs\helper;

// Paging options.
$page      = optional_param('page', 0, PARAM_INT);
$perpage   = optional_param('perpage', 20, PARAM_INT);
$sort      = optional_param('sort', 'course', PARAM_ALPHANUM);
$dir       = optional_param('dir', 'ASC', PARAM_ALPHA);
// Action.
$action    = optional_param('action', false, PARAM_ALPHA);
$search    = optional_param('search', '', PARAM_TEXT);

// Filters.
$filterschool = optional_param('filterschool', 0, PARAM_INT);
$filterbasecategory = optional_param('filterbasecategory', '', PARAM_TEXT);
$filtercategory = optional_param('filtercategory', '', PARAM_INT);
$filtercourse = optional_param('filtercourse', '', PARAM_INT);
$filtergroup = optional_param('filtergroup', '', PARAM_INT);
$filtercourserole = optional_param('filtercourserole', '', PARAM_INT);
$filtergraduation = optional_param('filtergraduation', '', PARAM_RAW);
$filterprogram = optional_param('filterprogram', '', PARAM_RAW);

$cfg = get_config('report_ghs');

require_login(null, false);
$contextsystem = context_system::instance();

// Permission.
if (!has_capability('report/ghs:viewenrollment', $contextsystem) && !\report_ghs\helper::has_capability_in_any_course('report/ghs:viewenrollment')) {
    throw new required_capability_exception($context, $capability, 'nopermissions', '');
}

$thispageurl = new moodle_url('/report/ghs/ghs_enrollment.php');
$fullpageurl = new moodle_url('/report/ghs/ghs_enrollment.php', [
    'page' => $page,
    'perpage' => $perpage,
    'sort' => $sort,
    'dir' => $dir,
    'action' => $action,
    'filterschool' => $filterschool,
    'filterbasecategory' => $filterbasecategory,
    'filtercategory' => $filtercategory,
    'filtercourse' => $filtercourse,
    'filtergroup' => $filtergroup,
    'filtercourserole' => $filtercourserole,
    'filtergraduation' => $filtergraduation,
    'filterprogram' => $filterprogram,
]);
$report = basename($thispageurl->out(), '.php');

$PAGE->set_url($thispageurl);
$PAGE->set_pagelayout('report');
$PAGE->set_context($contextsystem);
$PAGE->add_body_classes(['path-report-ghs', 'path-blocks-ned_teacher_tools']);

$name = get_string('ghsenrollment', 'report_ghs');
$title = get_string('ghsenrollment', 'report_ghs');
$heading = $SITE->fullname;

// Breadcrumb.
if (has_capability('report/ghs:viewenrollment', $contextsystem)) {
    admin_externalpage_setup('reportghsenrollment', null, null);
}

$PAGE->set_title($title);
$PAGE->set_heading($heading);

$datacolumns = array(
    'id' => 'r.id',
    'school' => 'coh.name',
    'cohortid' => 'r.cohortid',
    'basecategory' => 'cc.parent',
    'category' => 'cc.name',
    'categoryid' => 'cc.id',
    'groupid' => 'r.groupid',
    'groupname' => 'g.name',
    'userid' => 'u.id',
    'firstname' => 'u.firstname',
    'lastname' => 'u.lastname',
    'email' => 'u.email',
    'course' => 'c.shortname',
    'courseid' => 'c.id',
    'coursestart' => 'r.timestart',
    'courseend' => 'r.timeend',
    'groupstart' => 'g.startdate',
    'groupend' => 'g.enddate',
    'courserole' => 'ro.name',
    'courseroleid' => 'ro.id',
    'oen' => 'r.oen',
    'defaultrole' => 'r.defaultrole',
    'graduation' => 'r.graduation',
    'program' => 'r.program',
    'registered' => 'r.registered',
    'frate' => 'r.frate',
    'srate' => 'r.srate'
);

$params = [];

// Filter.
$where = '';
if ($filterschool) {
    $where .= " AND ".$datacolumns['cohortid']." = :school";
    $params['school'] = $filterschool;
}
if ($filterbasecategory) {
    $where .= " AND ".$datacolumns['basecategory']." = :basecategory";
    $params['basecategory'] = $filterbasecategory;
}
if ($filtercategory) {
    $where .= " AND ".$datacolumns['categoryid']." = :category";
    $params['category'] = $filtercategory;
}
if ($filtercourse) {
    $where .= " AND ".$datacolumns['courseid']." = :course";
    $params['course'] = $filtercourse;
}
if ($filtergroup) {
    $where .= " AND ".$datacolumns['groupid']." = :group";
    $params['group'] = $filtergroup;
}
if ($filtercourserole) {
    $where .= " AND ".$datacolumns['courseroleid']." = :courserole";
    $params['courserole'] = $filtercourserole;
}
if ($filtergraduation) {
    $where .= " AND ".$datacolumns['graduation']." = :graduation";
    $params['graduation'] = $filtergraduation;
}
if ($filterprogram) {
    $where .= " AND ".$datacolumns['program']." = :program";
    $params['program'] = $filterprogram;
}


// Sort.
$order = '';
if ($sort) {
    $order = " ORDER BY $datacolumns[$sort] $dir";
}

$courselist = helper::get_report_courses();

list($insql, $cparams) = $DB->get_in_or_equal($courselist);

$pageparams = array();
$pageparams['insql'] = $insql;
$pageparams['params'] = $cparams;

// Filter by capabilies.
\report_ghs\helper::report_filter($where, $params, $report);

// Count records for paging.
$countsql = "SELECT COUNT(1) 
FROM {report_ghs_enrollment} r
INNER JOIN {user} u ON r.userid = u.id
INNER JOIN {course} c ON r.courseid = c.id
INNER JOIN {course_categories} cc ON c.category = cc.id
LEFT OUTER JOIN {cohort} coh ON r.cohortid = coh.id
LEFT OUTER JOIN {groups} g ON r.groupid = g.id
INNER JOIN {role} ro ON r.courserole = ro.id
WHERE 0=0
$where";

$totalcount = $DB->count_records_sql($countsql, $params);

// Table columns.
$columns = array(
    'rowcount',
    'school',
    'basecategory',
    'category',
    'course',
    'groupname',
    'firstname',
    'lastname',
    'courserole',
    'coursestart',
    'courseend',
    'groupstart',
    'groupend',
    'graduation',
    'program',
    'frate',
    'srate',
);

$sql = "SELECT
r.id,
coh.name school,
cc.parent basecategory,
cc.name category,
r.groupid,
g.name groupname,
g.startdate groupstart,
g.enddate groupend,
u.id userid,
u.firstname,
u.lastname,
u.email,
c.shortname course,
c.id courseid,
r.timestart coursestart,
r.timeend courseend,
ro.name courserole,
r.oen,
r.defaultrole,
r.graduation,
r.program,
r.registered,
r.frate,
r.srate
FROM {report_ghs_enrollment} r
INNER JOIN {user} u ON r.userid = u.id
INNER JOIN {course} c ON r.courseid = c.id
INNER JOIN {course_categories} cc ON c.category = cc.id
LEFT OUTER JOIN {cohort} coh ON r.cohortid = coh.id
LEFT OUTER JOIN {groups} g ON r.groupid = g.id
INNER JOIN {role} ro ON r.courserole = ro.id
WHERE 0=0
               $where              
               $order";

if ($action == 'excel') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);

    $table = new stdClass();
    $table->head = $columns;

    // Delete first rowcount column.
    $itemid = array_shift($table->head);
    // Delete last action column.
    //array_pop($table->head);

    $counter = 0;
    $filename = $report.'_'.(date('Y-m-d'));
    $downloadfilename = clean_filename($filename);

    $workbook = new Spreadsheet();
    $myxls = $workbook->setActiveSheetIndex(0);

    $numberofcolumns = count($table->head);

    // Header row.
    foreach ($table->head as $key => $heading) {
        $cell = Coordinate::stringFromColumnIndex($key + 1) . '1'; // A1 cell address.
        $myxls->setCellValue($cell, str_replace("\n", ' ', htmlspecialchars_decode(strip_tags(nl2br(get_string($heading, 'report_ghs'))))));
    }

    // Data rows.
    $rs = $DB->get_recordset_sql($sql, $params);
    $rownum = 2;
    foreach ($rs as $tablerow) {
        $row = array();
        $columnum = 1;
        foreach ($table->head as $column) {
            $data = \report_ghs\helper::enrollment_data($tablerow, $column, $counter, $pageparams, true);
            $cell = Coordinate::stringFromColumnIndex($columnum) . $rownum; // A2 cell address.
            if (preg_match("/^[fh]tt?p:\/\//", $data)) {
                $myxls->setCellValue($cell, 'Link');
                $myxls->getCell($cell)->getHyperlink()->setUrl($data);
            } else {
                $myxls->setCellValue($cell, $data);
            }
            $columnum++;
        }
        $rownum++;
    }
    $rs->close();


    // Freeze header.
    $myxls->freezePane('A2');

    // Filter.
    $myxls->setAutoFilter(
        $myxls->calculateWorksheetDimension()
    );

    // Auto column width calculation.
    foreach (range('A', $myxls->getHighestDataColumn()) as $col) {
        $myxls->getColumnDimension($col)->setAutoSize(true);
    }

    // Header format.
    $styleArray = array(
        'font' => array(
            'bold' => true,
        ),
        'fill' => array(
            'fillType' => Fill::FILL_SOLID,
            'color' => array(
                'argb' => 'FFFFF000',
            ),
        ),
    );
    $myxls->getStyle('A1:'.Coordinate::stringFromColumnIndex($numberofcolumns).'1')->applyFromArray($styleArray);

    // Rename worksheet
    $myxls->setTitle('export');

    // Set active sheet index to the first sheet, so Excel opens this as the first sheet
    $workbook->setActiveSheetIndex(0);

    $objWriter = new Xlsx($workbook);

    if (ob_get_length()) {
        ob_end_clean();
    }
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment;filename="'.$downloadfilename.'.xlsx"');
    header('Cache-Control: max-age=0');

    ob_end_clean();
    $objWriter->save('php://output');
    exit;
} else if ($action == 'csv') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);
    $table = new stdClass();

    // Delete firs rowcount column.
    array_shift($columns);
    // Delete last action column.
    //array_pop($columns);

    $headers = $columns;

    foreach ($headers as $ckey => $column) {
        $headers[$ckey] = get_string($column, 'report_ghs');
    }

    if (ob_get_length()) {
        ob_end_clean();
    }
    // Output headers so that the file is downloaded rather than displayed.
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=ghs_enrollment_'.(date('Y-m-d')).'.csv');

    // Create a file pointer connected to the output stream.
    $outputcsv = fopen('php://output', 'w');

    // Output the column headings.
    fputcsv($outputcsv, $headers);

    $counter = 0;

    $rs = $DB->get_recordset_sql($sql, $params);
    foreach ($rs as $tablerow) {
        $row = array();
        foreach ($columns as $column) {
            $row[] = \report_ghs\helper::enrollment_data($tablerow, $column, $counter, $pageparams, true);
        }
        fputcsv($outputcsv, $row);
    }
    $rs->close();
    exit;
} else {

    foreach ($columns as $column) {
        $string[$column] = get_string($column, 'report_ghs');
        if ($sort != $column) {
            $columnicon = "";
            $columndir = "ASC";
        } else {
            $columndir = $dir == "ASC" ? "DESC" : "ASC";
            $columnicon = ($dir == "ASC") ? "sort_asc" : "sort_desc";
            $columnicon = $OUTPUT->pix_icon('t/'.$columnicon, '', 'moodle', array('class' => 'iconsort'));
        }
        if (($column == 'rowcount') || ($column == 'action') || ($column == 'gradebooklink') || ($column == 'kicagradebooklink')) {
            $$column = $string[$column];
        } else {
            $sorturl = clone $fullpageurl;
            $sorturl->param('perpage', $perpage);
            $sorturl->param('sort', $column);
            $sorturl->param('dir', $columndir);
            $sorturl->param('search', $search);

            $$column = html_writer::link($sorturl->out(false), $string[$column]).' '.$columnicon;
        }
    }

    $table = new html_table();

    $table->head = array();
    $table->wrap = array();
    $table->attributes = ['class' => 'nedtable fullwidth'];

    foreach ($columns as $column) {
        $table->head[$column] = $$column;
        $table->wrap[$column] = '';
    }

    // Override cell wrap.
    $table->wrap['action'] = 'nowrap';

    $tablerows = $DB->get_records_sql($sql, $params, $page * $perpage, $perpage);

    $counter = ($page * $perpage);

    foreach ($tablerows as $tablerow) {
        $row = new html_table_row();

        foreach ($columns as $column) {
            $varname = 'cell'.$column;
            $$varname = new html_table_cell(\report_ghs\helper::enrollment_data($tablerow, $column, $counter, $pageparams));
        }

        $row->cells = array();
        foreach ($columns as $column) {
            $varname = 'cell' . $column;
            $row->cells[$column] = $$varname;
        }
        $table->data[] = $row;

    }

    echo $OUTPUT->header();
    echo html_writer::start_div('page-content-wrapper', array('id' => 'page-content'));
    echo html_writer::tag('h1', $title, array('class' => 'page-title'));

    $reportcls = '\report_ghs\task\adhoc_'.$report.'_report';

    if (class_exists($reportcls)) {
        if ($task = $reportcls::get_task()) {
            echo html_writer::tag('div', get_string('taskwarning', 'report_ghs', userdate($task->customdata->timecreated)), array('class' => 'alert alert-info'));
        } else {
            $a = new stdClass();
            $a->lastupdate = userdate($cfg->enrollmentlastupdate);
            $a->updateurl = (new moodle_url('/report/ghs/update_report.php', ['report' => $report]))->out();
            echo html_writer::tag('div', get_string('lastupdatewarning', 'report_ghs', $a), array('class' => 'alert alert-info'));
        }
    }

    \report_ghs\helper::print_export_task_info($report, $cfg);

    // Filter form.
    $searchformurl = new moodle_url('/report/ghs/ghs_enrollment.php');

    // Schools
    $sql = "SELECT DISTINCT r.cohortid, c.name
              FROM {report_ghs_enrollment} r
              JOIN {cohort} c ON r.cohortid = c.id
             WHERE 0 = 0
          ORDER BY c.name ASC";

    $schooloptions = $DB->get_records_sql_menu($sql, $params);

    $basecategoryoptions = [
        '81' => '2019-20',
        '82' => '2018-19',
    ];

    $sql = "SELECT DISTINCT cc.id, cc.name
              FROM {report_ghs_enrollment} r
              JOIN {course} c ON r.courseid = c.id
              JOIN {course_categories} cc
                ON c.category = cc.id
             WHERE 0 = 0
          ORDER BY cc.name";
    $categoryoptions = $DB->get_records_sql_menu($sql, $params);

    $sql = "SELECT DISTINCT c.id, c.shortname
              FROM {report_ghs_enrollment} r
              JOIN {course} c ON r.courseid = c.id
             WHERE 0 = 0
          ORDER BY c.shortname";
    $courseoptions = $DB->get_records_sql_menu($sql, $params);

    $sql = "SELECT DISTINCT g.id, g.name
              FROM {report_ghs_enrollment} r
              JOIN {groups} g ON r.groupid = g.id
             WHERE 0 = 0
          ORDER BY g.name";
    $groupoptions = $DB->get_records_sql_menu($sql, $params);

    $sql = "SELECT DISTINCT g.id, g.name
              FROM {report_ghs_enrollment} r
              JOIN {role} g ON r.courserole = g.id
             WHERE 0 = 0
          ORDER BY g.name";
    $courseroleoptions = $DB->get_records_sql_menu($sql, $params);

    $sql = "SELECT DISTINCT r.graduation, r.graduation name
              FROM {report_ghs_enrollment} r
          ORDER BY r.graduation";
    $graduationoptions = $DB->get_records_sql_menu($sql, $params);

    $sql = "SELECT DISTINCT r.program, r.program name
              FROM {report_ghs_enrollment} r
          ORDER BY r.program";
    $programoptions = $DB->get_records_sql_menu($sql, $params);

    $searchform = html_writer::tag('form',
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'sesskey',
            'value' => sesskey(),
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'perpage',
            'value' => $perpage,
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'sort',
            'value' => $sort,
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'dir',
            'value' => $dir,
        )).

        // First row.
        html_writer::start_div('form-inline').
        html_writer::start_div('form-group').

        html_writer::label(get_string('school', 'report_ghs'), 'filterschool', true, ['class' => '']).
        html_writer::select(
            $schooloptions, 'filterschool', $filterschool, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('basecategory', 'report_ghs'), 'filterbasecategory', true, ['class' => '_sr-only']).
        html_writer::select(
            $basecategoryoptions, 'filterbasecategory', $filterbasecategory, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('category', 'report_ghs'), 'filtercategory', true, ['class' => 'ml-3']).
        html_writer::select(
            $categoryoptions, 'filtercategory', $filtercategory, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('course', 'report_ghs'), 'filtercourse', true, ['class' => 'ml-3']).
        html_writer::select(
            $courseoptions, 'filtercourse', $filtercourse, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::end_div().
        html_writer::end_div().

        // Second row.
        html_writer::start_div('form-inline').
        html_writer::start_div('form-group').

        html_writer::label(get_string('group', 'report_ghs'), 'filtergroup', true, ['class' => 'ml-3']).
        html_writer::select(
            $groupoptions, 'filtergroup', $filtergroup, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('courserole', 'report_ghs'), 'filtercourserole', true, ['class' => 'ml-3']).
        html_writer::select(
            $courseroleoptions, 'filtercourserole', $filtercourserole, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('graduation', 'report_ghs'), 'filtergraduation', true, ['class' => 'ml-3']).
        html_writer::select(
            $graduationoptions, 'filtergraduation', $filtergraduation, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('program', 'report_ghs'), 'filterprogram', true, ['class' => 'ml-3']).
        html_writer::select(
            $programoptions, 'filterprogram', $filterprogram, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::empty_tag('input', array(
            'type' => 'submit',
            'value' => get_string('filter', 'report_ghs'),
            'class' => 'btn btn-primary form-control mb-2 mr-sm-2',
        )).

        html_writer::link($searchformurl,
            get_string('reset', 'report_ghs'), array('class' => 'btn btn-secondary form-control mb-2 mr-sm-2 filter-reset')
        ).

        html_writer::end_div().
        html_writer::end_div(),
        array(
            'action' => $searchformurl->out(),
            'method' => 'post',
            'class' => '_form-inline mb-2 mt-2',
            'autocomplete' => 'off'
        )
    );

    echo html_writer::div($searchform, 'search-form-wrapper', array('id' => 'search-form'));

    $pagingurl = clone $fullpageurl;
    $pagingurl->remove_params('page');
    $pagingurl->remove_params('action');

    $pagingbar = new paging_bar($totalcount, $page, $perpage, $pagingurl, 'page');

    // Export link.
    $exporexcelurl = $thispageurl;
    $exporexcelurl->remove_all_params();
    $exporexcelurl->param('action', 'excel');
    $exportexceliconurl = $OUTPUT->image_url('f/spreadsheet');
    $exportexcelicon = html_writer::img($exportexceliconurl, '', array('width' => '16', 'height' => '16'));
    $exportexceliconlink = html_writer::link($exporexcelurl, $exportexcelicon);
    $exportexcellink = html_writer::link($exporexcelurl, 'XLS');

    $exportexturl = $thispageurl;
    $exportexturl->remove_all_params();
    $exportexturl->param('action', 'csv');
    $exporttexticonurl = $OUTPUT->image_url('f/text');
    $exporttexticon = html_writer::img($exporttexticonurl, '', array('width' => '16', 'height' => '16'));
    $exporttexticonlink = html_writer::link($exportexturl, $exporttexticon);
    $exporttextlink = html_writer::link($exportexturl, 'CSV');

    $exportbuttons =  html_writer::div(
        get_string('export', 'report_ghs').' :&nbsp;&nbsp;'.
        $exportexceliconlink.'&nbsp;'.$exportexcellink.'&nbsp;|&nbsp; '
        .$exporttexticonlink.'&nbsp;'.$exporttextlink, 'export-link-wrapper',
        array('style' => 'text-align:center;')
    );

    if ($outputpagingbar = $OUTPUT->render($pagingbar)) {
        echo $outputpagingbar;
    } else {
        echo html_writer::tag('div', '', ['class' => 'dummy-pagination']);
    }
    echo html_writer::div($exportbuttons, 'export-link-wrapper-top');
    echo html_writer::table($table);
    echo $outputpagingbar;
    echo $exportbuttons;

    echo html_writer::end_div(); // Main wrapper.
    echo $OUTPUT->footer();
}