<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * RADIX
 *
 * @package     report_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once(dirname(dirname(dirname(__FILE__))) . '/config.php');
require_once("$CFG->libdir/phpspreadsheet/vendor/autoload.php");
require_once($CFG->libdir.'/adminlib.php');

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Fill;

// Paging options.
$page      = optional_param('page', 0, PARAM_INT);
$perpage   = optional_param('perpage', 250, PARAM_INT);
$sort      = optional_param('sort', '', PARAM_ALPHANUM);
$dir       = optional_param('dir', 'ASC', PARAM_ALPHA);
$export    = optional_param('export', 'full', PARAM_ALPHA);
// Action.
$action    = optional_param('action', false, PARAM_ALPHA);
$search    = optional_param('search', '', PARAM_TEXT);

// Filters.
$filterschool = optional_param('filterschool', 0, PARAM_INT);
$filterdepartment = optional_param('filterdepartment', '', PARAM_TEXT);
$filteractivestudents = optional_param('filteractivestudents', '', PARAM_TEXT);
$filterstartdate = optional_param('filterstartdate', 0, PARAM_INT);
$filterenddate = optional_param('filterenddate', 0, PARAM_INT);
$filterincludeclassesnotactive = optional_param('includeclassesnotactive', false, PARAM_BOOL);
$filterincludenoncreditcourses = optional_param('includenoncreditcourses', false, PARAM_BOOL);

$cfg = get_config('report_ghs');

require_login(null, false);
$contextsystem = context_system::instance();

// Role
$isot = report_ghs\helper::is_ot($USER->id);

// Permission.
if (!has_capability('report/ghs:viewgroupenrollment', $contextsystem) && !\report_ghs\helper::has_capability_in_any_course('report/ghs:viewgroupenrollment')) {
    throw new required_capability_exception($context, $capability, 'nopermissions', '');
}

$thispageurl = new moodle_url('/report/ghs/ghs_group_enrollment.php');
$fullpageurl = new moodle_url('/report/ghs/ghs_group_enrollment.php', [
    'page' => $page,
    'perpage' => $perpage,
    'sort' => $sort,
    'dir' => $dir,
    'action' => $action,
    'search' => $search,
    'filterschool' => $filterschool,
    'filterdepartment' => $filterdepartment,
    'filteractivestudents' => $filteractivestudents,
    'filterstartdate' => $filterstartdate,
    'filterenddate' => $filterenddate,
    'includeclassesnotactive' => $filterincludeclassesnotactive,
    'includenoncreditcourses' => $filterincludenoncreditcourses,
]);
$report = basename($thispageurl->out(), '.php');

$PAGE->set_url($thispageurl);
$PAGE->set_pagelayout('report');
$PAGE->set_context($contextsystem);
$PAGE->add_body_classes(['path-report-ghs', 'path-blocks-ned_teacher_tools']);

$name = get_string('ghsgroupenrollment', 'report_ghs');
$title = get_string('ghsgroupenrollment', 'report_ghs');
$heading = $title;

// Breadcrumb.
if (has_capability('report/ghs:viewgroupenrollment', $contextsystem)) {
    admin_externalpage_setup('reportghsgroupenrollment', null, null);
}

$PAGE->set_title($title);
$PAGE->set_heading($heading);

$datacolumns = array(
    'activestudents' => 'r.activestudents',
    'cohortid' => 'r.cohortid',
    'course' => 'c.fullname',
    'path' => 'cc.path',
    'coursecode' => 'c.shortname',
    'ccategory' => 'c.category',
    'courseid' => 'r.courseid',
    'ctname' => "if(r.ctid = -1, '*multiple', (SELECT u3.firstname FROM {user} u3 WHERE u3.id = r.ctid))",
    'department' => 'r.department',
    'dmdateconflicts' => 'r.dmdateconflicts',
    'dmrequired' => 'g.schedule',
    'dmstatus' => 'r.dmstatus',
    'enddate' => 'g.enddate',
    'enrolldateconflicts' => 'r.enrolldateconflicts',
    'gmessaging' => '(SELECT COUNT(1) FROM {message_conversations} conv WHERE conv.type = \'2\' AND conv.itemid = g.id AND conv.enabled = \'1\')',
    'grade' => 'r.grade',
    'groupid' => 'r.groupid',
    'groupidnumber' => 'g.idnumber',
    'groupname' => 'g.name',
    'id' => 'r.id',
    'moe_code' => 'r.moe_code',
    'moe_name' => 'r.moe_name',
    'otid' => 'r.otid',
    'otname' => "if(r.otid = -1, '*multiple', (SELECT u3.firstname FROM {user} u3 WHERE u3.id = r.otid))",
    'school' => ' coh.name',
    'startdate' => 'g.startdate',
    'subject' => 'r.subject',
    'suspendedstudents' => 'r.suspendedstudents',
    'totaldays' => '(CEIL((g.enddate - g.startdate) / 86400))',
);

$params = [];

// Filter.
$where = '';
if ($isot) {
    $where .= " AND ".$datacolumns['otid']." = :otid";
    $params['otid'] = $USER->id;
}
if ($search) {
    $where .= " AND (" . $DB->sql_like($datacolumns['course'], ':search1', false, false);
    $params['search1'] = '%'.$DB->sql_like_escape($search).'%';

    $where .= " OR " . $DB->sql_like($datacolumns['coursecode'], ':search2', false, false);
    $params['search2'] = '%'.$DB->sql_like_escape($search).'%';

    $where .= " OR " . $DB->sql_like($datacolumns['school'], ':search3', false, false);
    $params['search3'] = '%'.$DB->sql_like_escape($search).'%';

    $where .= " OR " . $DB->sql_like($datacolumns['groupname'], ':search4', false, false);
    $params['search4'] = '%'.$DB->sql_like_escape($search).'%';

    $where .= " OR " . $DB->sql_like($datacolumns['ctname'], ':search5', false, false);
    $params['search5'] = '%'.$DB->sql_like_escape($search).'%';

    $where .= " OR " . $DB->sql_like($datacolumns['otname'], ':search6', false, false) . ")";
    $params['search6'] = '%'.$DB->sql_like_escape($search).'%';
}
if ($filterschool) {
    if ($filterschool > 0) {
        $where .= " AND " . $datacolumns['cohortid'] . " = :school";
        $params['school'] = $filterschool;
    } else {
        $where .= " AND " . $datacolumns['cohortid'] . " != :school";
        $params['school'] = -$filterschool;
    }
}
if ($filterdepartment) {
    $where .= " AND ".$datacolumns['department']." = :department";
    $params['department'] = $filterdepartment;
}
if (isset($filteractivestudents) && $filteractivestudents != '') {
    $where .= " AND ".$datacolumns['activestudents']." != 0";
    switch ($filteractivestudents) {
        case '0':
            $where .= " AND {$datacolumns['activestudents']} = 0";
            break;
        case '1':
            $where .= " AND {$datacolumns['activestudents']} = 1";
            break;
        case '1+':
            $where .= " AND {$datacolumns['activestudents']} > 1";
            break;
        case '10+':
            $where .= " AND {$datacolumns['activestudents']} > 10";
            break;
    }
}
if ($filterstartdate) {
    switch ($filterstartdate) {
        case 1: // None.
            $where .= " AND ({$datacolumns['startdate']} = 0 OR {$datacolumns['startdate']} IS NULL)";
            break;
        case 2: // Past.
            $where .= " AND ".$datacolumns['startdate']." < " .time() . " AND ({$datacolumns['startdate']} != 0 AND {$datacolumns['startdate']} IS NOT NULL)";
            break;
        case 3: // Future
            $where .= " AND ".$datacolumns['startdate']." > " .time() . " AND ({$datacolumns['startdate']} != 0 AND {$datacolumns['startdate']} IS NOT NULL)";;
            break;
    }
}
if ($filterenddate) {
    switch ($filterenddate) {
        case 1: // None.
            $where .= " AND ({$datacolumns['enddate']} = 0 OR {$datacolumns['enddate']} IS NULL)";
            break;
        case 2: // Past.
            $where .= " AND ".$datacolumns['enddate']." < " .time() . " AND ({$datacolumns['enddate']} != 0 AND {$datacolumns['enddate']} IS NOT NULL)";;
            break;
        case 3: // Future
            $where .= " AND ".$datacolumns['enddate']." > " .time() . " AND ({$datacolumns['enddate']} != 0 AND {$datacolumns['enddate']} IS NOT NULL)";;
            break;
    }
}
if (!$filterincludeclassesnotactive) {
    $time = time();
    $where .= " AND ({$datacolumns['startdate']} < :time1 AND :time2 < {$datacolumns['enddate']})";
    $params['time1'] = $time;
    $params['time2'] = $time;
}
if (!$filterincludenoncreditcourses) {
    $where .= " AND {$datacolumns['path']} NOT LIKE '/110/%'";
}

// Sort.
$order = '';
if ($sort) {
    $order = " ORDER BY $datacolumns[$sort] $dir";
} else {
    $order = " ORDER BY timecheck DESC, {$datacolumns['enddate']} ASC";
}

$pageparams = array();

// Filter by capabilies.
\report_ghs\helper::report_filter($where, $params, $report, 'report/ghs:viewgroupenrollment');

// Count records for paging.
$countsql = "SELECT COUNT(1)
            FROM {report_ghs_group_enrollment} r
          JOIN {course} c
            ON r.courseid = c.id
          JOIN {course_categories} cc 
            ON c.category = cc.id                
          JOIN {groups} g
            ON r.groupid = g.id
LEFT OUTER JOIN {cohort} coh ON r.cohortid = coh.id
         WHERE 0 = 0 
               $where";
$totalcount = $DB->count_records_sql($countsql, $params);

// Table columns.
$columns = array(
    'rowcount',
    'course',
    'coursecode',
    //'subject',
    //'grade',
    //'department',
    //'moe_code',
    'school',
    'groupname',
    //'gmessaging',
    'activestudents',
    //'suspendedstudents',
    'ctname',
    'otname',
    'startdate',
    'enddate',
    'totaldays',
    //'dmrequired',
    'dmstatus'
);
$columnsexport = array(
    'rowcount',
    'course',
    'coursecode',
    'subject',
    'grade',
    'department',
    'moe_code',
    'category',
    'basecategory',
    'school',
    'groupname',
    'groupid',
    'groupidnumber',
    'gmessaging',
    'activestudents',
    'suspendedstudents',
    'ctname',
    'otname',
    'startdate',
    'enddate',
    'totaldays',
    'dmrequired',
    'dmstatus'
);

$sql = "SELECT r.id,
               c.fullname course,
               c.shortname coursecode,
               coh.name school,
               g.name groupname,
               g.idnumber groupidnumber,
               r.activestudents,
               r.suspendedstudents,
               if(r.ctid = -1, '*multiple', (SELECT u3.firstname FROM {user} u3 WHERE u3.id = r.ctid)) ctname,
               if(r.otid = -1, '*multiple', (SELECT u3.firstname FROM {user} u3 WHERE u3.id = r.otid)) otname,
               g.startdate,
               g.enddate,
               (CEIL((g.enddate - g.startdate) / 86400)) totaldays,
               g.schedule dmrequired,
               r.dmdateconflicts,
               r.dmstatus,
               r.enrolldateconflicts,
               r.courseid,
               r.groupid,
               cc.name category,
               cc.path path,
               cc.parent basecategory,
               (SELECT COUNT(1) FROM {message_conversations} conv WHERE conv.type = '2' AND conv.itemid = g.id AND conv.enabled = '1') gmessaging,
               r.subject,
               r.grade,
               r.department,
               r.moe_code,
               r.moe_name,
               (SELECT e.id FROM {enrol} e WHERE e.courseid = r.courseid AND e.enrol = 'manual' AND e.status = '0') enrolid,
               (CASE WHEN g.enddate > ".time()." THEN \"1\" ELSE \"0\" END) timecheck
          FROM {report_ghs_group_enrollment} r
          JOIN {course} c
            ON r.courseid = c.id
          JOIN {course_categories} cc 
            ON c.category = cc.id                
          JOIN {groups} g
            ON r.groupid = g.id
LEFT OUTER JOIN {cohort} coh ON r.cohortid = coh.id
         WHERE 0 = 0
               $where
               $order";

if ($action == 'excel') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);

    $table = new stdClass();
    $table->head = ($export == 'current') ? $columns : $columnsexport;

    // Delete first rowcount column.
    $itemid = array_shift($table->head);
    // Delete last action column.
    //array_pop($table->head);

    $counter = 0;
    $filename = $report.'_'.(date('Y-m-d'));
    $downloadfilename = clean_filename($filename);

    $workbook = new Spreadsheet();
    $myxls = $workbook->setActiveSheetIndex(0);

    $numberofcolumns = count($table->head);

    $gradecolumns = array('coursegrade', 'kicaavg', 'kica70', 'kica30');

    // Header row.
    foreach ($table->head as $key => $heading) {
        $cell = Coordinate::stringFromColumnIndex($key + 1) . '1'; // A1 cell address.
        $myxls->setCellValue($cell, str_replace("\n", ' ', htmlspecialchars_decode(strip_tags(nl2br(get_string($heading, 'report_ghs'))))));
    }

    // Data rows.
    $rs = $DB->get_recordset_sql($sql, $params);
    $rownum = 2;
    foreach ($rs as $tablerow) {
        $row = array();
        $columnum = 1;
        $col = [];
        foreach ($table->head as $column) {
            $data = \report_ghs\helper::group_enrollment_data($tablerow, $column, $counter, $pageparams, true);
            $col[$column] = ['index' => $columnum, 'value' => $data];

            $lowgrade  = false;
            if (in_array($column, $gradecolumns) && is_numeric($data) && $data < 50)  {
                $lowgrade  = true;
            }

            $cell = Coordinate::stringFromColumnIndex($columnum) . $rownum; // A2 cell address.
            if (preg_match("/^[fh]tt?ps?:\/\//", $data)) {
                $linktext = \report_ghs\helper::group_enrollment_data($tablerow, $column.'_txt', $counter, $pageparams, true);
                $myxls->setCellValue($cell, $linktext);
                $myxls->getCell($cell)->getHyperlink()->setUrl($data);
            } else {
                $myxls->setCellValue($cell, $data);
            }

            if ($lowgrade) {
                $myxls->getStyle($cell)->getFill()->setFillType(Fill::FILL_SOLID);
                $myxls->getStyle($cell)->getFill()->getStartColor()->setARGB('FFFCC7CE');
                $myxls->getStyle($cell)->getFont()->getColor()->setARGB('FF9C0006');
            }

            $columnum++;
        }

        if (is_numeric($col['coursegrade']['value']) && is_numeric($col['kica70']['value'])) {
            if ($col['coursegrade']['value'] > 50 && $col['kica70']['value'] < 50) {
                $cell = Coordinate::stringFromColumnIndex($col['coursegrade']['index']) . $rownum;
                $myxls->getStyle($cell)->getFill()->setFillType(Fill::FILL_SOLID);
                $myxls->getStyle($cell)->getFill()->getStartColor()->setARGB('FFC0514D');
                $myxls->getStyle($cell)->getFont()->getColor()->setARGB('FFFFFFFF');

                $cell = Coordinate::stringFromColumnIndex($col['kica70']['index']) . $rownum;
                $myxls->getStyle($cell)->getFill()->setFillType(Fill::FILL_SOLID);
                $myxls->getStyle($cell)->getFill()->getStartColor()->setARGB('FFC0514D');
                $myxls->getStyle($cell)->getFont()->getColor()->setARGB('FFFFFFFF');
            }
        }

        $rownum++;
    }
    $rs->close();

    // Freeze header.
    $myxls->freezePane('A2');

    // Filter.
    $myxls->setAutoFilter(
        $myxls->calculateWorksheetDimension()
    );

    // Auto column width calculation.
    foreach (range('A', $myxls->getHighestDataColumn()) as $col) {
        $myxls->getColumnDimension($col)->setAutoSize(true);
    }

    // Header format.
    $styleArray = array(
        'font' => array(
            'bold' => true,
        ),
        'fill' => array(
            'fillType' => Fill::FILL_SOLID,
            'color' => array(
                'argb' => 'FFFFF000',
            ),
        ),
    );
    $myxls->getStyle('A1:'.Coordinate::stringFromColumnIndex($numberofcolumns).'1')->applyFromArray($styleArray);

    // Rename worksheet
    $myxls->setTitle('export');

    // Set active sheet index to the first sheet, so Excel opens this as the first sheet
    $workbook->setActiveSheetIndex(0);

    $objWriter = new Xlsx($workbook);

    if (ob_get_length()) {
        ob_end_clean();
    }
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment;filename="'.$downloadfilename.'.xlsx"');
    header('Cache-Control: max-age=0');

    ob_end_clean();
    $objWriter->save('php://output');
    exit;
} else if ($action == 'csv') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);
    $table = new stdClass();

    // Delete firs rowcount column.
    array_shift($columnsexport);
    // Delete last action column.
    //array_pop($columnsexport);

    $headers = $columnsexport;

    foreach ($headers as $ckey => $column) {
        $headers[$ckey] = get_string($column, 'report_ghs');
    }

    if (ob_get_length()) {
        ob_end_clean();
    }
    // Output headers so that the file is downloaded rather than displayed.
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename='.$report.'_'.(date('Y-m-d')).'.csv');

    // Create a file pointer connected to the output stream.
    $outputcsv = fopen('php://output', 'w');

    // Output the column headings.
    fputcsv($outputcsv, $headers);

    $counter = 0;

    $rs = $DB->get_recordset_sql($sql, $params);
    foreach ($rs as $tablerow) {
        $row = array();
        foreach ($columnsexport as $column) {
            $row[] = \report_ghs\helper::group_enrollment_data($tablerow, $column, $counter, $pageparams, true);
        }
        fputcsv($outputcsv, $row);
    }
    $rs->close();
    exit;
} else {

    foreach ($columns as $column) {
        $string[$column] = get_string($column, 'report_ghs');
        if ($sort != $column) {
            $columnicon = "";
            $columndir = "ASC";
        } else {
            $columndir = $dir == "ASC" ? "DESC" : "ASC";
            $columnicon = ($dir == "ASC") ? "sort_asc" : "sort_desc";
            $columnicon = $OUTPUT->pix_icon('t/'.$columnicon, '', 'moodle', array('class' => 'iconsort'));
        }
        if (($column == 'rowcount') || ($column == 'action') || ($column == 'kicadiff') || ($column == 'kicalink')) {
            $$column = $string[$column];
        } else {
            $sorturl = clone $fullpageurl;
            $sorturl->param('sort', $column);
            $sorturl->param('dir', $columndir);

            $$column = html_writer::link($sorturl->out(false), $string[$column]).' '.$columnicon;
        }
    }

    $table = new html_table();

    $table->head = array();
    $table->wrap = array();
    $table->attributes = ['class' => 'nedtable fullwidth'];

    foreach ($columns as $column) {
        $table->head[$column] = $$column;
        $table->wrap[$column] = '';
    }

    // Override cell wrap.
    $table->wrap['action'] = 'nowrap';
    $table->wrap['dmstatus'] = 'nowrap';

    $tablerows = $DB->get_records_sql($sql, $params, $page * $perpage, $perpage);

    $counter = ($page * $perpage);

    foreach ($tablerows as $tablerow) {
        $row = new html_table_row();

        foreach ($columns as $column) {
            $varname = 'cell'.$column;
            $$varname = new html_table_cell(\report_ghs\helper::group_enrollment_data($tablerow, $column, $counter, $pageparams));

            if (isset($tablerow->{$column.'cls'})) {
                $$varname->attributes['class'] = 'bg-alert';
            }
        }

        $row->cells = array();
        foreach ($columns as $column) {
            $varname = 'cell' . $column;
            $row->cells[$column] = $$varname;
        }
        $table->data[] = $row;

    }

    echo $OUTPUT->header();
    echo html_writer::start_div('page-content-wrapper', array('id' => 'page-content'));
    //echo html_writer::tag('h1', $title, array('class' => 'page-title'));
    echo html_writer::div(
        html_writer::link(new moodle_url('/index.php'), get_string('close', 'report_ghs'), ['class' => 'btn btn-secondary btn-close']),
        'btn-close-wrapper'
    );

    $collapsibleoutput = '';

    $reportcls = '\report_ghs\task\adhoc_'.$report.'_report';

    if (class_exists($reportcls)) {
        if ($task = $reportcls::get_task()) {
            $collapsibleoutput .= html_writer::tag('div', get_string('taskwarning', 'report_ghs', userdate($task->customdata->timecreated)), array('class' => 'alert alert-info'));
        } else {
            $a = new stdClass();
            $a->lastupdate = userdate($cfg->groupenrollmentlastupdate);
            $a->updateurl = (new moodle_url('/report/ghs/update_report.php', ['report' => $report]))->out();
            $collapsibleoutput .= html_writer::tag('div', get_string('lastupdatewarning', 'report_ghs', $a), array('class' => 'alert alert-info'));
        }
    }

    $collapsibleoutput .= report_ghs\helper::print_export_task_info($report, $cfg, true);


    // Filter form.
    $searchformurl = new moodle_url('/report/ghs/ghs_group_enrollment.php');

    if (has_capability('report/ghs:groupenrollmentfilter', $contextsystem)) {
        $searchform = html_writer::tag('form',
            html_writer::empty_tag('input', array('type' => 'hidden', 'name' => 'sesskey', 'value' => sesskey())) .
            html_writer::empty_tag('input', array('type' => 'hidden', 'name' => 'perpage', 'value' => $perpage)) .
            html_writer::empty_tag('input', array('type' => 'hidden', 'name' => 'sort', 'value' => $sort)) .
            html_writer::empty_tag('input', array('type' => 'hidden', 'name' => 'dir', 'value' => $dir)) .
            html_writer::empty_tag('input', array('type' => 'hidden', 'name' => 'filterschool', 'value' => $filterschool)) .
            html_writer::empty_tag('input', array('type' => 'hidden', 'name' => 'filterdepartment', 'value' => $filterdepartment)) .
            html_writer::empty_tag('input', array('type' => 'hidden', 'name' => 'filteractivestudents', 'value' => $filteractivestudents)) .
            html_writer::empty_tag('input', array('type' => 'hidden', 'name' => 'filterstartdate', 'value' => $filterstartdate)) .
            html_writer::empty_tag('input', array('type' => 'hidden', 'name' => 'filterenddate', 'value' => $filterenddate)) .

            // First row.
            html_writer::start_div('form-inline') .
            html_writer::start_div('form-group', ['style' => 'margin:10px auto 0']) .

            html_writer::div(
                html_writer::empty_tag('input', ['type' => 'text', 'class' => 'form-control',
                    'placeholder'=> get_string('search'), 'name' => 'search', 'value' => $search]).
                html_writer::empty_tag('input', ['type' => 'submit', 'class' => 'btn btn-primary ml-2 color-green', 'name' => 'btnsubmit', 'value' => get_string('go')]),
                'form-group'
            ).
            html_writer::div(
                html_writer::link($searchformurl,
                    get_string('reset', 'report_ghs'), array('class' => 'btn btn-secondary form-control mb-2 mr-sm-2 filter-reset')
                ),
                'input-group',
                ['style' => 'padding-left: 10px;margin-top: 8px;']
            ).

            html_writer::end_div().
            html_writer::end_div().

            html_writer::start_div('form-row') .
            html_writer::start_div('form-group', ['style' => 'margin:10px auto 0']) .

            html_writer::div(
                html_writer::checkbox('includeclassesnotactive', 1, $filterincludeclassesnotactive, get_string('includeclassesnotactive', 'report_ghs'),
                    ['class' => 'form-check-input', 'id' => 'includeclassesnotactive', 'onChange' => 'this.form.submit()']),
                'form-check'
            ).
            html_writer::div(
                html_writer::checkbox('includenoncreditcourses', 1, $filterincludenoncreditcourses, get_string('includenoncreditcourses', 'report_ghs'),
                    ['class' => 'form-check-input', 'id' => 'includenoncreditcourses', 'onChange' => 'this.form.submit()']),
                'form-check'
            ).

            html_writer::end_div().
            html_writer::end_div(),
            array(
                'action' => $searchformurl->out(),
                'method' => 'post',
                'class' => '_form-inline mb-2 mt-2',
                'autocomplete' => 'off'
            )
        );
        echo html_writer::div($searchform, 'search-form-wrapper', array('id' => 'search-form'));

        // Schools
        $sql = "SELECT DISTINCT r.cohortid, c.name
                FROM {report_ghs_group_enrollment} r
                JOIN {cohort} c ON r.cohortid = c.id
               WHERE 0 = 0
             ORDER BY c.name ASC";

        $schooloptions = $DB->get_records_sql_menu($sql, $params);
        if ($independent = array_search('Independent', $schooloptions)){
            $schooloptions = [-$independent => 'Hide independent'] + $schooloptions;
        }

        $sql = "SELECT DISTINCT r.department, r.department name
              FROM {report_ghs_group_enrollment} r
          ORDER BY r.department";

        $departmentoptions = $DB->get_records_sql_menu($sql, $params);

        $includeactivestudentsoptions = [
            '' => get_string('any'),
            '0' => '0',
            '1' => '1',
            '1+' => '1+',
            '10+' => '10+',
        ];

        $startdateoptions = $enddateoptions = [
            '1' => get_string('none'),
            '2' => get_string('past', 'report_ghs'),
            '3' => get_string('future', 'report_ghs'),
        ];

        $searchform = html_writer::tag('form',
            html_writer::empty_tag('input', array(
                'type' => 'hidden',
                'name' => 'sesskey',
                'value' => sesskey(),
            )) .
            html_writer::empty_tag('input', array(
                'type' => 'hidden',
                'name' => 'perpage',
                'value' => $perpage,
            )) .
            html_writer::empty_tag('input', array(
                'type' => 'hidden',
                'name' => 'sort',
                'value' => $sort,
            )) .
            html_writer::empty_tag('input', array(
                'type' => 'hidden',
                'name' => 'dir',
                'value' => $dir,
            )) .

            // First row.
            html_writer::start_div('form-inline') .
            html_writer::start_div('form-group') .

            html_writer::label(get_string('school', 'report_ghs'), 'filterschool', true, ['class' => '']) .
            html_writer::select(
                $schooloptions, 'filterschool', $filterschool, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
            ) .

            html_writer::label(get_string('department', 'report_ghs'), 'filterdepartment', true, ['class' => '']) .
            html_writer::select(
                $departmentoptions, 'filterdepartment', $filterdepartment, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
            ) .

            html_writer::label(get_string('startdate', 'report_ghs'), 'filterstartdate', true, ['class' => 'ml-3']).
            html_writer::select(
                $startdateoptions, 'filterstartdate', $filterstartdate, array('' => get_string('any')), ['class' => 'form-control mb-2 mr-sm-2']
            ).

            html_writer::label(get_string('enddate', 'report_ghs'), 'filterenddate', true, ['class' => 'ml-3']).
            html_writer::select(
                $enddateoptions, 'filterenddate', $filterenddate, array('' => get_string('any')), ['class' => 'form-control mb-2 mr-sm-2']
            ).

            html_writer::label(get_string('activestudents', 'report_ghs'), 'filteractivestudents', true, ['class' => 'ml-3']).
            html_writer::select(
                $includeactivestudentsoptions, 'filteractivestudents', $filteractivestudents, false, ['class' => 'form-control mb-2 mr-sm-2']
            ).

            html_writer::empty_tag('input', array(
                'type' => 'submit',
                'value' => get_string('filter', 'report_ghs'),
                'class' => 'btn btn-primary form-control mb-2 mr-sm-2',
            )) .

            html_writer::link($searchformurl,
                get_string('reset', 'report_ghs'), array('class' => 'btn btn-secondary form-control mb-2 mr-sm-2 filter-reset')
            ) .

            html_writer::end_div() .
            html_writer::end_div(),
            array(
                'action' => $searchformurl->out(),
                'method' => 'post',
                'class' => '_form-inline mb-2 mt-2',
                'autocomplete' => 'off'
            )
        );

        $collapsibleoutput .= html_writer::div($searchform, 'search-form-wrapper', array('id' => 'search-form'));

    } else {
        $searchform = html_writer::tag('form',
            html_writer::empty_tag('input', array('type' => 'hidden', 'name' => 'sesskey', 'value' => sesskey())) .
            html_writer::empty_tag('input', array('type' => 'hidden', 'name' => 'perpage', 'value' => $perpage,)) .
            html_writer::empty_tag('input', array('type' => 'hidden', 'name' => 'sort', 'value' => $sort)) .
            html_writer::empty_tag('input', array('type' => 'hidden', 'name' => 'dir', 'value' => $dir)) .

            // First row.
            html_writer::start_div('form-inline') .
            html_writer::start_div('form-group', ['style' => 'margin:10px auto 0']) .

            html_writer::div(
                html_writer::div(
                    html_writer::empty_tag('input', ['type' => 'text', 'class' => 'form-control',
                        'placeholder'=> get_string('search'), 'name' => 'search', 'value' => $search]).
                    html_writer::empty_tag('input', ['type' => 'submit', 'class' => 'btn btn-primary ml-2 color-green', 'name' => 'btnsubmit', 'value' => get_string('go')]),
                    'form-group'
                ),
                'input-group'
            ).

            html_writer::div(
                html_writer::link($searchformurl,
                    get_string('reset', 'report_ghs'), array('class' => 'btn btn-secondary form-control mb-2 mr-sm-2 filter-reset')
                ),
                'input-group',
                ['style' => 'padding-left: 10px;margin-top: 8px;']
            ).

            html_writer::end_div().
            html_writer::end_div().

            html_writer::start_div('form-row') .
            html_writer::start_div('form-group', ['style' => 'margin:10px auto 0']) .

            html_writer::div(
                html_writer::checkbox('includeclassesnotactive', '1', $filterincludeclassesnotactive, get_string('includeclassesnotactive', 'report_ghs'),
                    ['class' => 'form-check-input', 'id' => 'includeclassesnotactive', 'onChange' => 'this.form.submit()']),
                'form-check'
            ).
            html_writer::div(
                html_writer::checkbox('includenoncreditcourses', '1', $filterincludenoncreditcourses, get_string('includenoncreditcourses', 'report_ghs'),
                    ['class' => 'form-check-input', 'id' => 'includenoncreditcourses', 'onChange' => 'this.form.submit()']),
                'form-check'
            ).

            html_writer::end_div().
            html_writer::end_div(),

            array(
                'action' => $searchformurl->out(),
                'method' => 'post',
                'class' => '_form-inline mb-2 mt-2',
                'autocomplete' => 'off'
            )
        );
        echo html_writer::div($searchform, 'search-form-wrapper', array('id' => 'search-form'));
    }

    // Export link.
    if (has_capability('report/ghs:groupenrollmentexport', $contextsystem)) {
        $exporexcelurl = clone $thispageurl;
        $exporexcelurl->remove_all_params();
        $exporexcelurl->param('action', 'excel');
        $exportexceliconurl = $OUTPUT->image_url('f/spreadsheet');
        $exportexcelicon = html_writer::img($exportexceliconurl, '', array('width' => '16', 'height' => '16'));
        $exportexceliconlink = html_writer::link($exporexcelurl, $exportexcelicon);
        $exportexcellink = html_writer::link($exporexcelurl, 'XLS');

        $exportexturl = clone $thispageurl;
        $exportexturl->remove_all_params();
        $exportexturl->param('action', 'csv');
        $exporttexticonurl = $OUTPUT->image_url('f/text');
        $exporttexticon = html_writer::img($exporttexticonurl, '', array('width' => '16', 'height' => '16'));
        $exporttexticonlink = html_writer::link($exportexturl, $exporttexticon);
        $exporttextlink = html_writer::link($exportexturl, 'CSV');

        $exportbuttons = html_writer::div(
            get_string('exportfullreport', 'report_ghs') . ' :&nbsp;&nbsp;' .
            $exportexceliconlink . '&nbsp;' . $exportexcellink . '&nbsp;|&nbsp; '
            . $exporttexticonlink . '&nbsp;' . $exporttextlink, 'export-link-wrapper',
            array('style' => 'text-align:right;')
        );

        // Export current view.
        $exporexcelurl->param('export', 'current');
        $exportexceliconlink = html_writer::link($exporexcelurl, $exportexcelicon);
        $exportexcellink = html_writer::link($exporexcelurl, 'XLS');

        $exportexturl->param('export', 'current');
        $exporttexticonlink = html_writer::link($exportexturl, $exporttexticon);
        $exporttextlink = html_writer::link($exportexturl, 'CSV');

        $exportbuttonscurrent = html_writer::div(
            get_string('exportcurrentview', 'report_ghs') . ' :&nbsp;&nbsp;' .
            $exportexceliconlink . '&nbsp;' . $exportexcellink . '&nbsp;|&nbsp; '
            . $exporttexticonlink . '&nbsp;' . $exporttextlink, 'export-link-wrapper',
            array('style' => 'text-align:right;')
        );

        $uploadclassenrollment = '';
        if (has_capability('report/ghs:uploadgroupenrollment', $contextsystem)) {
            $uploadclassenrollment = html_writer::div(
                html_writer::link(
                    new moodle_url('/report/ghs/import.php'),
                    '<i class="fa fa-upload" aria-hidden="true"></i> ' . get_string('uploadgroupenrollment', 'report_ghs')
                ),
                'import-link-wrapper-top text-right'
            );
        }

        $collapsibleoutput .= html_writer::div($exportbuttons . $exportbuttonscurrent . $uploadclassenrollment, 'export-link-wrapper-top');
    }

    if ($collapsibleoutput) {
        if (has_capability('report/ghs:viewadvancedoptions', $contextsystem)) {
            print_collapsible_region(
                $collapsibleoutput,
                'collapsible-wrapper',
                'id-collapsible-wrapper',
                get_string('advancedoptions', 'report_ghs'),
                '',
                true
            );
        }

        echo html_writer::tag('div ', get_string('lastupdatewarningshort', 'report_ghs', userdate($cfg->groupenrollmentlastupdate)),
            ['class' => 'last-update-warning-short text-right']);
    }

    $pagingurl = clone $fullpageurl;
    $pagingurl->remove_params('page');
    $pagingurl->remove_params('action');

    $pagingbar = new paging_bar($totalcount, $page, $perpage, $pagingurl, 'page');

    if ($outputpagingbar = $OUTPUT->render($pagingbar)) {
        echo $outputpagingbar;
    } else {
        echo html_writer::tag('div', '', ['class' => 'dummy-pagination']);
    }

    echo html_writer::table($table);
    echo $outputpagingbar ?? '';
    echo $exportbuttons ?? '';

    echo html_writer::end_div(); // Main wrapper.
    echo $OUTPUT->footer();
}