<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * RADIX
 *
 * @package     report_ghs
 * @copyright   2020 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once(dirname(dirname(dirname(__FILE__))) . '/config.php');
require_once("$CFG->libdir/phpspreadsheet/vendor/autoload.php");
require_once($CFG->libdir.'/adminlib.php');

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Fill;

// Paging options.
$page      = optional_param('page', 0, PARAM_INT);
$perpage   = optional_param('perpage', 100, PARAM_INT);
$sort      = optional_param('sort', 'course', PARAM_ALPHANUM);
$dir       = optional_param('dir', 'ASC', PARAM_ALPHA);
// Action.
$action    = optional_param('action', false, PARAM_ALPHA);
$search    = optional_param('search', '', PARAM_TEXT);

// Filters.
$strsuspendedcourse = get_string('suspendedcourse', 'report_ghs');
$strsuspendedsite = get_string('suspendedsite', 'report_ghs');
$stractive = get_string('active', 'report_ghs');
$strhide = get_string('hide', 'report_ghs');
$strshow = get_string('show', 'report_ghs');

$filterbasecategory = optional_param('filterbasecategory', 81, PARAM_INT);
$filtergcomp = optional_param('filtergcomp', '1-99', PARAM_TEXT);
$filtergrade = optional_param('filtergrade', 'none', PARAM_TEXT);
$filterhasduedate = optional_param('filterhasduedate', '', PARAM_TEXT);
$filterschool = optional_param('filterschool', 0, PARAM_INT);
$filtercourse = optional_param('filtercourse', 0, PARAM_INT);
$filteruserstatus = optional_param('filteruserstatus', $stractive, PARAM_TEXT);
$filterexcluded = optional_param('filterexcluded', $strhide, PARAM_TEXT);
$filtercompleted = optional_param('filtercompleted', $strhide, PARAM_TEXT);

$cfg = get_config('report_ghs');

require_login(null, false);
$contextsystem = context_system::instance();

// Role
$isot = report_ghs\helper::is_ot($USER->id);

// Permission.
if (!$isot) {
    if (!has_capability('report/ghs:viewgrades', $contextsystem) && !\report_ghs\helper::has_capability_in_any_course('report/ghs:viewgrades')) {
        throw new required_capability_exception($context, $capability, 'nopermissions', '');
    }
}

$thispageurl = new moodle_url('/report/ghs/ghs_missed_grade.php');
$fullpageurl = new moodle_url('/report/ghs/ghs_missed_grade.php', [
    'page' => $page,
    'perpage' => $perpage,
    'sort' => $sort,
    'dir' => $dir,
    'action' => $action,
    'filtergcomp' => $filtergcomp,
    'filtergrade' => $filtergrade,
    'filterhasduedate' => $filterhasduedate,
    'filterschool' => $filterschool,
    'filtercourse' => $filtercourse,
    'filteruserstatus' => $filteruserstatus,
    'filterexcluded' => $filterexcluded,
    'filtercompleted' => $filtercompleted,
    'filterbasecategory' => $filterbasecategory,
]);
$report = basename($thispageurl->out(), '.php');

$PAGE->set_url($thispageurl);
$PAGE->set_pagelayout('report');
$PAGE->set_context($contextsystem);
$PAGE->add_body_classes(['path-report-ghs', 'path-blocks-ned_teacher_tools']);

$name = get_string('ghsmissedgrade', 'report_ghs');
$title = get_string('ghsmissedgrade', 'report_ghs');
$heading = $SITE->fullname;

// Breadcrumb.
if (has_capability('report/ghs:viewgrades', $contextsystem)) {
    admin_externalpage_setup('reportghsmissedgrade', null, null);
}

$PAGE->set_title($title);
$PAGE->set_heading($heading);

$datacolumns = array(
    'id' => 'r.id',
    'firstname' => 'u.firstname',
    'lastname' => 'u.lastname',
    'school' => 'coh.name',
    'course' => 'cor.shortname',
    'category' => 'cat.name',
    'basecategory' => 'cat.parent',
    'groupname' => 'g.name',
    'ctname' => "if(r.ctid = -1, '*multiple', (SELECT u3.firstname FROM {user} u3 WHERE u3.id = r.ctid))",
    'otname' => "if(r.otid = -1, '*multiple', (SELECT u3.firstname FROM {user} u3 WHERE u3.id = r.otid))",
    'daysoverdue' => "if(DATEDIFF(CURDATE(), FROM_UNIXTIME(r.duedate)) > 0, DATEDIFF(CURDATE(), FROM_UNIXTIME(r.duedate)), null)",
    'duedate' => 'r.duedate',
    'activityname' => 'r.activityname',
    'activitygrade' => 'r.activitygrade',
    'kicagrade' => 'r.kicagrade',
    'skipped' => 'r.skipped',
    'gcomp' => 'r.gcomp',
    'gcomprate' => '(r.graded/r.all*100)',
    'chortid' => 'r.chortid',
    'courseid' => 'r.courseid',
    'userstatus' => 'r.userstatus',
    'excluded' => 'r.excluded',
    'completed' => 'r.completed',
);

$params = [];

// Filter.
$where = '';

if ($filterbasecategory) {
    $where .= " AND ".$datacolumns['basecategory']." = :basecategory";
    $params['basecategory'] = $filterbasecategory;
}

if ($filtergcomp != '') {
    if ($filtergcomp === '1-99') {
        $where .= " AND {$datacolumns['gcomprate']} > 0 AND {$datacolumns['gcomprate']} < 100";
    } else if ($filtergcomp === '0') {
        $where .= " AND {$datacolumns['gcomprate']} = 0";
    } else if ($filtergcomp === '100') {
        $where .= " AND {$datacolumns['gcomprate']} = 100";
    }
}

if ($filtergrade != '') {
    if ($filtergrade === 'none') {
        $where .= " AND ({$datacolumns['activitygrade']} IS NULL OR {$datacolumns['kicagrade']} IS NULL)";
    } else if ($filtergrade === 'zero') {
        $where .= " AND ({$datacolumns['activitygrade']} = 0 OR {$datacolumns['kicagrade']} = 0)";
    }
}

if ($filterhasduedate != '') {
    if ($filterhasduedate === 'yes') {
        $where .= " AND {$datacolumns['duedate']} > 0";
    } else if ($filterhasduedate === 'no') {
        $where .= " AND ({$datacolumns['duedate']} = 0 OR {$datacolumns['duedate']} IS NULL)";
    }
}

if ($filterschool) {
    $where .= " AND ".$datacolumns['chortid']." = :school";
    $params['school'] = $filterschool;
}

if ($filtercourse) {
    $where .= " AND ".$datacolumns['courseid']." = :course";
    $params['course'] = $filtercourse;
}

if ($filteruserstatus) {
    $where .= " AND ".$datacolumns['userstatus']." = :userstatus";
    $params['userstatus'] = $filteruserstatus;
}

if ($filterexcluded == $strhide) {
    $where .= " AND ".$datacolumns['excluded']." = :excluded";
    $params['excluded'] = 0;
}

if ($filtercompleted == $strhide) {
    $where .= " AND ".$datacolumns['completed']." = :completed";
    $params['completed'] = 0;
}

// Sort.
$order = '';
if ($sort) {
    $order = " ORDER BY $datacolumns[$sort] $dir";
}

$pageparams = array();

// Filter by capabilies.
// \report_ghs\helper::report_filter($where, $params, $report);

// Count records for paging.
$countsql = "SELECT COUNT(1)
               FROM {report_ghs_missed_grade} r
         INNER JOIN {user} u ON r.userid = u.id
    LEFT OUTER JOIN {cohort} coh ON r.chortid = coh.id
    LEFT OUTER JOIN {course} cor ON r.courseid = cor.id
    LEFT OUTER JOIN {course_categories} cat ON cor.category = cat.id
    LEFT OUTER JOIN {groups} g ON r.groupid = g.id 
              WHERE 0 = 0 $where";
$totalcount = $DB->count_records_sql($countsql, $params);

// Table columns.
$columns = array(
    'rowcount',
    'firstname',
    'lastname',
    'school',
    'course',
    'groupname',
    'ctname',
    'otname',
    'duedate',
    'daysoverdue',
    'activityname',
    'activitygrade',
    'kicagrade',
    'skipped',
    'gcomp',
);
$columnsexport = array(
    'rowcount',
    'firstname',
    'lastname',
    'school',
    'userstatus',
    'excluded',
    'completed',
    'basecategory',
    'course',
    'groupname',
    'ctname',
    'otname',
    'duedate',
    'daysoverdue',
    'activityname',
    'activitygrade',
    'kicagrade',
    'skipped',
    'gcomp',
);

$sql = "SELECT  r.*,
                u.firstname,
                u.lastname,
                coh.name  school,
                cor.shortname  course,
                cat.name  category,
                g.name  groupname,
                cat.parent basecategory,       
                if(r.ctid = -1, '*multiple', (SELECT u3.firstname FROM {user} u3 WHERE u3.id = r.ctid)) ctname,
                if(r.otid = -1, '*multiple', (SELECT u3.firstname FROM {user} u3 WHERE u3.id = r.otid)) otname,
                if(DATEDIFF(CURDATE(), FROM_UNIXTIME(r.duedate)) > 0, DATEDIFF(CURDATE(), FROM_UNIXTIME(r.duedate)), null) daysoverdue       
           FROM {report_ghs_missed_grade} r
     INNER JOIN {user} u ON r.userid = u.id
LEFT OUTER JOIN {cohort} coh ON r.chortid = coh.id
LEFT OUTER JOIN {course} cor ON r.courseid = cor.id
LEFT OUTER JOIN {course_categories} cat ON cor.category = cat.id
LEFT OUTER JOIN {groups} g ON r.groupid = g.id
          WHERE 0 = 0
                $where
                $order";

if ($action == 'excel') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);

    $table = new stdClass();
    $table->head = $columnsexport;

    // Delete first rowcount column.
    $itemid = array_shift($table->head);
    // Delete last action column.
    //array_pop($table->head);


    /*$rs = $DB->get_recordset_sql($sql, $params);
    foreach ($rs as $tablerow) {
        $row = array();
        $rowclass = array();
        foreach ($table->head as $column) {
            $row[] = \report_ghs\helper::missed_grade_data($tablerow, $column, $counter, $pageparams, true);
            $rowclass[] = \report_ghs\helper::missed_grade_class($tablerow, $column, $counter, $pageparams, true);
        }
        $table->data[] = $row;
        $table->dataclass[] = $rowclass;
    }
    $rs->close();*/

    $counter = 0;
    $filename = $report.'_'.(date('Y-m-d'));
    $downloadfilename = clean_filename($filename);

    $workbook = new Spreadsheet();
    $myxls = $workbook->setActiveSheetIndex(0);

    $numberofcolumns = count($table->head);

    $styleOrange = array(
        'fill' => array(
            'type' => Fill::FILL_SOLID,
            'color' => array('rgb' => 'FFA500'),
        ),
    );

    $styleRed = array(
        'font' => array(
            'color' => array('rgb' => 'FFFFFF'),
        ),
        'fill' => array(
            'type' => Fill::FILL_SOLID,
            'color' => array('rgb' => 'FF0000'),
        ),
    );

    // Header row.
    foreach ($table->head as $key => $heading) {
        $cell = Coordinate::stringFromColumnIndex($key + 1) . '1'; // A1 cell address.
        $myxls->setCellValue($cell, str_replace("\n", ' ', htmlspecialchars_decode(strip_tags(nl2br(get_string($heading, 'report_ghs'))))));
    }

    // Data rows.
    $rs = $DB->get_recordset_sql($sql, $params);
    $rownum = 2;
    foreach ($rs as $tablerow) {
        $row = array();
        $columnum = 1;
        foreach ($table->head as $column) {
            $data = \report_ghs\helper::missed_grade_data($tablerow, $column, $counter, $pageparams, true);
            $class = \report_ghs\helper::missed_grade_class($tablerow, $column, $counter, $pageparams, true);
            $cell = Coordinate::stringFromColumnIndex($columnum) . $rownum; // A2 cell address.

            if ($class == 'bg-orange') {
                $workbook->getActiveSheet()->getStyle($cell)->applyFromArray($styleOrange);
            } else if ($class == 'bg-red') {
                $workbook->getActiveSheet()->getStyle($cell)->applyFromArray($styleRed);
            }

            if (preg_match("/^[fh]tt?p:\/\//", $data)) {
                $myxls->setCellValue($cell, 'Link');
                $myxls->getCell($cell)->getHyperlink()->setUrl($data);
            } else {
                $myxls->setCellValue($cell, $data);
            }
            $columnum++;
        }
        $rownum++;
    }
    $rs->close();

    // Freeze header.
    $myxls->freezePane('A2');

    // Filter.
    $myxls->setAutoFilter(
        $myxls->calculateWorksheetDimension()
    );

    // Auto column width calculation.
    foreach (range('A', $myxls->getHighestDataColumn()) as $col) {
        $myxls->getColumnDimension($col)->setAutoSize(true);
    }

    // Header format.
    $styleArray = array(
        'font' => array(
            'bold' => true,
        ),
        'fill' => array(
            'fillType' => Fill::FILL_SOLID,
            'color' => array(
                'argb' => 'FFFFF000',
            ),
        ),
    );
    $myxls->getStyle('A1:'.Coordinate::stringFromColumnIndex($numberofcolumns).'1')->applyFromArray($styleArray);

    // Rename worksheet
    $myxls->setTitle('export');

    // Set active sheet index to the first sheet, so Excel opens this as the first sheet
    $workbook->setActiveSheetIndex(0);

    $objWriter = new Xlsx($workbook);

    if (ob_get_length()) {
        ob_end_clean();
    }
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment;filename="'.$downloadfilename.'.xlsx"');
    header('Cache-Control: max-age=0');

    ob_end_clean();
    $objWriter->save('php://output');
    exit;
} else if ($action == 'csv') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);
    $table = new stdClass();

    // Delete firs rowcount column.
    array_shift($columnsexport);
    // Delete last action column.
    //array_pop($columnsexport);

    $headers = $columnsexport;

    foreach ($headers as $ckey => $column) {
        $headers[$ckey] = get_string($column, 'report_ghs');
    }

    if (ob_get_length()) {
        ob_end_clean();
    }
    // Output headers so that the file is downloaded rather than displayed.
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=ghs_missed_grade_'.(date('Y-m-d')).'.csv');

    // Create a file pointer connected to the output stream.
    $outputcsv = fopen('php://output', 'w');

    // Output the column headings.
    fputcsv($outputcsv, $headers);

    $counter = 0;

    $rs = $DB->get_recordset_sql($sql, $params);
    foreach ($rs as $tablerow) {
        $row = array();
        foreach ($columnsexport as $column) {
            $row[] = \report_ghs\helper::missed_grade_data($tablerow, $column, $counter, $pageparams, true);
        }
        fputcsv($outputcsv, $row);
    }
    $rs->close();
    exit;
} else {

    foreach ($columns as $column) {
        $string[$column] = get_string($column, 'report_ghs');
        if ($sort != $column) {
            $columnicon = "";
            $columndir = "ASC";
        } else {
            $columndir = $dir == "ASC" ? "DESC" : "ASC";
            $columnicon = ($dir == "ASC") ? "sort_asc" : "sort_desc";
            $columnicon = $OUTPUT->pix_icon('t/'.$columnicon, '', 'moodle', array('class' => 'iconsort'));
        }
        if (($column == 'rowcount') || ($column == 'action') || ($column == 'kicadiff') || ($column == 'kicalink')) {
            $$column = $string[$column];
        } else {
            $sorturl = clone $fullpageurl;
            $sorturl->remove_params('page');
            $sorturl->param('sort', $column);
            $sorturl->param('dir', $columndir);
            $$column = html_writer::link($sorturl->out(false), $string[$column]).' '.$columnicon;
        }
    }

    $table = new html_table();

    $table->head = array();
    $table->wrap = array();
    $table->attributes = ['class' => 'nedtable fullwidth'];

    foreach ($columns as $column) {
        $table->head[$column] = $$column;
        $table->wrap[$column] = '';
    }

    // Override cell wrap.
    $table->wrap['action'] = 'nowrap';

    $tablerows = $DB->get_records_sql($sql, $params, $page * $perpage, $perpage);

    $counter = ($page * $perpage);

    foreach ($tablerows as $tablerow) {
        $row = new html_table_row();

        foreach ($columns as $column) {
            $varname = 'cell'.$column;
            $$varname = new html_table_cell(\report_ghs\helper::missed_grade_data($tablerow, $column, $counter, $pageparams));
            $$varname->attributes['class'] = \report_ghs\helper::missed_grade_class($tablerow, $column, $counter, $pageparams);
        }

        $row->cells = array();
        foreach ($columns as $column) {
            $varname = 'cell' . $column;
            $row->cells[$column] = $$varname;
        }
        $table->data[] = $row;

    }

    echo $OUTPUT->header();
    echo html_writer::start_div('page-content-wrapper', array('id' => 'page-content'));
    echo html_writer::tag('h1', $title, array('class' => 'page-title'));

    $reportcls = '\report_ghs\task\adhoc_'.$report.'_report';

    if (class_exists($reportcls)) {
        if ($task = $reportcls::get_task()) {
            echo html_writer::tag('div', get_string('taskwarning', 'report_ghs', userdate($task->customdata->timecreated)), array('class' => 'alert alert-info'));
        } else {
            $a = new stdClass();
            $a->lastupdate = userdate($cfg->kicalastupdate);
            $a->updateurl = (new moodle_url('/report/ghs/update_report.php', ['report' => $report]))->out();
            echo html_writer::tag('div', get_string('lastupdatewarning', 'report_ghs', $a), array('class' => 'alert alert-info'));
        }
    }

    \report_ghs\helper::print_export_task_info($report, $cfg);

    // Form options.
    $basecategoryoptions = [
        '81' => '2019-20',
        '82' => '2018-19',
    ];

    $gcompoptions = [
        '1-99' => get_string('between199', 'report_ghs'),
        '0' => '0%',
        '100' => '100%',
    ];

    $gradeoptions = [
        'none' => get_string('none', 'report_ghs'),
        'zero' => get_string('zero', 'report_ghs'),
    ];

    $hasduedateoptions = [
        'yes' => get_string('yes'),
        'no' => get_string('no'),
    ];

    $sql = "SELECT DISTINCT r.chortid, c.name
              FROM {report_ghs_missed_grade} r
              JOIN {cohort} c ON r.chortid = c.id
             WHERE 0 = 0
          ORDER BY c.name ASC";

    $schooloptions = $DB->get_records_sql_menu($sql, $params);

    $sql = "SELECT DISTINCT r.courseid, c.shortname
              FROM {report_ghs_missed_grade} r
              JOIN {course} c ON r.courseid = c.id
             WHERE 0 = 0
          ORDER BY c.shortname ASC";

    $courseoptions = $DB->get_records_sql_menu($sql, $params);

    $userstatusoptions = [
        $stractive => $stractive,
        $strsuspendedsite => $strsuspendedsite,
        $strsuspendedcourse => $strsuspendedcourse,
    ];

    $excludedoptions = [
        $strshow => $strshow,
        $strhide => $strhide,
    ];

    $completedoptions = [
        $strshow => $strshow,
        $strhide => $strhide,
    ];

    // Filter form.
    $searchformurl = new moodle_url('/report/ghs/ghs_missed_grade.php');

    $searchform = html_writer::tag('form',
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'sesskey',
            'value' => sesskey(),
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'perpage',
            'value' => $perpage,
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'sort',
            'value' => $sort,
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'dir',
            'value' => $dir,
        )).

        // First row.
        html_writer::start_div('form-inline').
        html_writer::start_div('form-group').

        html_writer::label(get_string('basecategory', 'report_ghs'), 'filterbasecategory', true, ['class' => '_sr-only']).
        html_writer::select(
            $basecategoryoptions, 'filterbasecategory', $filterbasecategory, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('gcomp', 'report_ghs'), 'filtergcomp', true, ['class' => '_sr-only']).
        html_writer::select(
            $gcompoptions, 'filtergcomp', $filtergcomp, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('grade', 'report_ghs'), 'filtergrade', true, ['class' => '_sr-only']).
        html_writer::select(
            $gradeoptions, 'filtergrade', $filtergrade, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('hasduedate', 'report_ghs'), 'filterhasduedate', true, ['class' => '_sr-only']).
        html_writer::select(
            $hasduedateoptions, 'filterhasduedate', $filterhasduedate, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('school', 'report_ghs'), 'filterschool', true, ['class' => '_sr-only']).
        html_writer::select(
            $schooloptions, 'filterschool', $filterschool, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('course', 'report_ghs'), 'filtercourse', true, ['class' => '_sr-only']).
        html_writer::select(
            $courseoptions, 'filtercourse', $filtercourse, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::end_div().
        html_writer::start_div('form-group').

        html_writer::label(get_string('userstatus', 'report_ghs'), 'filteruserstatus', true, ['class' => '_sr-only']).
        html_writer::select(
            $userstatusoptions, 'filteruserstatus', $filteruserstatus, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('excludedgrade', 'report_ghs'), 'filterexcluded', true, ['class' => '_sr-only']).
        html_writer::select(
            $excludedoptions, 'filterexcluded', $filterexcluded, null, ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('completed', 'report_ghs'), 'filtercompleted', true, ['class' => '_sr-only']).
        html_writer::select(
            $completedoptions, 'filtercompleted', $filtercompleted, null, ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::empty_tag('input', array(
            'type' => 'submit',
            'value' => get_string('filter', 'report_ghs'),
            'class' => 'btn btn-primary form-control mb-2 mr-sm-2',
        )).

        html_writer::link($searchformurl,
            get_string('reset', 'report_ghs'), array('class' => 'btn btn-secondary form-control mb-2 mr-sm-2 filter-reset')
        ).


        html_writer::end_div().
        html_writer::end_div(),
        array(
            'action' => $searchformurl->out(),
            'method' => 'post',
            'class' => '_form-inline mb-2 mt-2',
            'autocomplete' => 'off'
        )
    );

    echo html_writer::div($searchform, 'search-form-wrapper', array('id' => 'search-form'));

    $pagingurl = clone $fullpageurl;
    $pagingurl->remove_params('page');
    $pagingurl->remove_params('action');

    $pagingbar = new paging_bar($totalcount, $page, $perpage, $pagingurl, 'page');

    // Export link.
    $exporexcelurl = clone $fullpageurl;
    $exporexcelurl->param('action', 'excel');
    $exportexceliconurl = $OUTPUT->image_url('f/spreadsheet');
    $exportexcelicon = html_writer::img($exportexceliconurl, '', array('width' => '16', 'height' => '16'));
    $exportexceliconlink = html_writer::link($exporexcelurl, $exportexcelicon);
    $exportexcellink = html_writer::link($exporexcelurl, 'XLS');

    $exportexturl = clone $fullpageurl;
    $exportexturl->param('action', 'csv');
    $exporttexticonurl = $OUTPUT->image_url('f/text');
    $exporttexticon = html_writer::img($exporttexticonurl, '', array('width' => '16', 'height' => '16'));
    $exporttexticonlink = html_writer::link($exportexturl, $exporttexticon);
    $exporttextlink = html_writer::link($exportexturl, 'CSV');

    $exportbuttons =  html_writer::div(
        get_string('export', 'report_ghs').' :&nbsp;&nbsp;'.
        $exportexceliconlink.'&nbsp;'.$exportexcellink.'&nbsp;|&nbsp; '
        .$exporttexticonlink.'&nbsp;'.$exporttextlink, 'export-link-wrapper',
        array('style' => 'text-align:center;')
    );

    if ($outputpagingbar = $OUTPUT->render($pagingbar)) {
        echo $outputpagingbar;
    } else {
        echo html_writer::tag('div', '', ['class' => 'dummy-pagination']);
    }
    echo html_writer::div($exportbuttons, 'export-link-wrapper-top');
    echo html_writer::table($table);
    echo $outputpagingbar;
    echo $exportbuttons;

    echo html_writer::end_div(); // Main wrapper.
    echo $OUTPUT->footer();
}