<?php
/**
 * Boost dashboard page
 *
 * @package    theme_ned_boost
 * @copyright  NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @noinspection PhpUnused
 */

namespace theme_ned_boost\output;
use block_ned_student_menu as SM;
use block_ned_teacher_tools as TT;
use local_kica as kica;
use local_ned_controller\base_empty_class;
use local_ned_controller\deadline_notification as DN;
use local_schoolmanager as LSM;
use local_schoolmanager\output\school_manager_render as LSMR;
use theme_ned_boost\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot.'/cohort/lib.php');
require_once($CFG->dirroot.'/calendar/lib.php');

/**
 * @property-read \context $ctx;
 * @property-read int $capability = self::CAP_CANT_SEE_DASHBOARD;
 * @property-read bool $has_kica
 * @property-read bool $has_schoolmanager
 * @property-read int $default_role_field_id
 * @property-read string $default_role_default_value
 * @property-read bool $show_inactive
 * @property-read int $course_type;
 * @property-read array $course_types = [];$ctx
 * @property-read \stdClass $viewer; // who see dashboard
 * @property-read int $viewerid = 0;
 * @property-read \stdClass $user; // who own dashboard
 * @property-read int $userid
 * @property-read array $schools = [];
 * @property-read \stdClass $school = null;
 * @property-read array $school_users = [];
 * @property-read int $chosen_schoolid = 0;
 * @property-read int $chosen_userid = 0;
 * @property-read \stdClass[] $course_records = [];
 * @property-read course[]  $courses = [];
 * @property-read \stdClass[] $groups_by_courseid = [];
 */
class frontdashboard implements \renderable, \templatable {
    use base_empty_class;

    const OTHER_PLUGINS = [
        'kica' => NED::KICA,
        'schoolmanager' => NED::SchM,
    ];

    const COLOR_GREEN = '#23b14c';
    const COLOR_BLUE = '#2f97de';
    const COLOR_LIGHT_ORANGE = '#f59c00';
    const COLOR_ORANGE = '#ea6b14';
    const COLOR_PURPLE = '#893bc3';
    const COLOR_GREY = '#c3c3c3';

    const KICA_COLORS = [
        'knowledge'     => self::COLOR_GREEN,
        'inquiry'       => self::COLOR_ORANGE,
        'communication' => self::COLOR_PURPLE,
        'application'   => self::COLOR_BLUE,
    ];

    const FIELD_ROLE = 'default_role';
    const FIELD_ED_PROGRAM = 'ed_program';
    const FIELD_GRADUATION = 'graduation';
    const USER_FIELDS = [self::FIELD_ROLE, self::FIELD_ED_PROGRAM, self::FIELD_GRADUATION];

    const PAGETYPE_PEOPLE = 'people';
    const PAGETYPE_DN = 'dn';
    const PAGETYPES = [self::PAGETYPE_PEOPLE, self::PAGETYPE_DN];

    const FIELD_COURSE_TYPE = 'course_type';

    const P_SHOWINACTIVE = 'showinactive';
    const P_USERID = 'userid';
    const P_SCHOOLID = 'schoolid';
    const P_COURSETYPE = 'coursetype';
    const P_PAGETYPE = 'pagetype';

    /*
     * Capabilities to see own or other dashboards.
     * All next includes others, so you can check by using ">" or "<"
     */
    const CAP_CANT_SEE_DASHBOARD = 0;
    const CAP_SEE_OWN_DASHBOARD = 1;
    const CAP_SEE_OWN_GROUPS_DASHBOARDS = 2;
    const CAP_SEE_OWN_COHORT_DASHBOARDS = 3;
    const CAP_SEE_ALL_DASHBOARDS = 4;

    const PAGE_MAIN = '/';
    const PAGE_DASHBOARD = '/my/';
    const PAGE_DASHBOARD_FULL = '/my/index.php';

    const COURSES_FOR_GRAPHS = 'Credit';

    static protected $_course_types = [];
    static protected $_default_role_field_id = null;
    static protected $_default_role_default_value = null;

    protected $_ctx;
    protected $_capability = self::CAP_CANT_SEE_DASHBOARD;
    protected $_has_kica = false;
    protected $_has_schoolmanager = false;
    protected $_show_inactive = false;
    protected $_pagetype = null;
    protected $_course_type = null;
    protected $_viewer; // who see dashboard
    protected $_viewerid = 0;
    protected $_user; // who own dashboard
    protected $_userid = 0;
    protected $_schools = [];
    /** @var \stdClass $_school */
    protected $_school = null;
    protected $_school_users = [];
    protected $_chosen_schoolid = 0;
    protected $_chosen_userid = 0;
    /** @var course[] $courses */
    protected $_courses = [];
    /** @var \stdClass[] $groups_by_courseid */
    protected $_groups_by_courseid = [];

    /** @var dashboard_content $c */
    public $c;
    /** @var \theme_ned_boost\output\core_renderer $o */
    public $o;

    public function __construct(){
        global $OUTPUT, $USER;
        $this->o = $OUTPUT;
        $this->c = new dashboard_content();

        $this->_ctx = \context_system::instance();
        $this->_capability = static::get_dashboard_capability($this->_ctx);
        if ($this->_capability < static::CAP_SEE_OWN_DASHBOARD){
            if ($this->o->page->url->get_path() != '/'){
                redirect(new \moodle_url(static::PAGE_MAIN));
            }
            return;
        }

        $this->_viewer = $USER;
        $this->_viewerid = $USER->id;
        static::_load_static_data();

        $this->_load_libs();
        $this->_get_schools();
        $this->_get_user_and_his_school();
        $this->_load_course_types();
        $this->_load_courses();
        $this->_load_groups();
        $this->_load_pagetypes();
    }

    /**
     * load static class data
     */
    static protected function _load_static_data(){
        static $done = false;

        if ($done){
            return;
        }

        global $DB;

        if (is_null(static::$_default_role_field_id)){
            $record = $DB->get_record('user_info_field', ['shortname' => static::FIELD_ROLE], 'id, defaultdata');
            static::$_default_role_field_id = $record->id ?? false;
            static::$_default_role_default_value = $record->defaultdata ?? '';
        }

        $done = true;
    }

    /**
     * load all possible fot current user schools from cohorts
     */
    protected function _get_schools(){
        if ($this->_capability >= static::CAP_SEE_ALL_DASHBOARDS){
            $cohort_data = cohort_get_all_cohorts(0, 0);
            $cohorts = $cohort_data['cohorts'] ?? [];
        } else {
            $cohorts = cohort_get_user_cohorts($this->_viewerid);
        }

        $this->_schools = static::get_schools_by_cohorts($cohorts);
    }

    /**
     * load chosen user, it school, and other users in this school
     */
    protected function _get_user_and_his_school(){
        if (empty($this->_schools) || $this->_capability <= static::CAP_SEE_OWN_DASHBOARD){
            $this->_chosen_schoolid = null;
            $this->_chosen_userid = null;
            $this->_school = reset($this->_schools);
            $this->_userid = $this->_viewerid;
            $this->_user = $this->_viewer;
            $this->_school_users = [];
            return;
        }

        if (count($this->_schools) == 1){
            $schoolid = key($this->_schools) ?: 0;
        } else {
            $schoolid = optional_param(static::P_SCHOOLID, 0, PARAM_INT);
            $schoolid = isset($this->_schools[$schoolid]) ? $schoolid : 0;
        }

        $school_users = static::get_school_members($schoolid ?: array_keys($this->_schools), NED::ROLE_STUDENT);
        if ($this->_capability < static::CAP_SEE_OWN_COHORT_DASHBOARDS){
            $school_users_ids = array_keys($school_users);
            $school_users_ids = NED::get_grader_students($this->_viewerid, null, null, $school_users_ids,
                true);
            $school_users = array_intersect_key($school_users, $school_users_ids);
        }

        $this->_school_users = $school_users;
        $userid = optional_param(static::P_USERID, 0, PARAM_INT);
        $userid = isset($school_users[$userid]) ? $userid : 0;


        if (!$userid || $userid == $this->_viewerid){
            $this->_user = $this->_viewer;
            $this->_school = $schoolid ? $this->_schools[$schoolid] : reset($this->_schools);
        } else {
            $this->_user = \core_user::get_user($userid);
            $cohorts = cohort_get_user_cohorts($userid);
            foreach ($cohorts as $id => $cohort){
                if (isset($this->_schools[$id])){
                    $this->_school = $this->_schools[$id];
                    break;
                }
            }
        }

        if ($this->_user && !isset($this->_user->profile)){
            $this->_user->profile = (array)profile_user_record($this->_user->id, false);
        }

        if ($this->_school && $this->_has_schoolmanager){
            $SM = LSM\school_manager::get_school_manager();
            $sm_school = $SM->get_school_by_ids($this->_school->id, true) ?: [];
            foreach ($sm_school as $key => $item){
                $this->_school->$key = $item;
            }
        }
        $this->_userid = $this->_user->id;

        $this->_chosen_userid = $userid;
        $this->_chosen_schoolid = $schoolid;
    }

    /**
     * load all course types
     */
    protected function _load_course_types(){
        $all_course_types = static::get_course_types();
        $course_dashboard_type_keys = explode(',', NED::get_config('dashboard_course_categories'));
        foreach ($course_dashboard_type_keys as $course_type_key){
            if (isset($all_course_types[$course_type_key])){
                static::$_course_types[$course_type_key] = $all_course_types[$course_type_key];
            }
        }

        $ct = optional_param(static::P_COURSETYPE, 0, PARAM_INT);
        if (!isset(static::$_course_types[$ct])){
            $ct = key(static::$_course_types);
        }
        $this->_course_type = $ct;
    }

    /**
     * load all courses for current user
     */
    protected function _load_courses(){
        $this->_show_inactive = optional_param(static::P_SHOWINACTIVE, false, PARAM_BOOL);
        $course_records = enrol_get_users_courses($this->userid, !$this->_show_inactive,'*');
        foreach ($course_records as $cr){
            $course = new course($cr);
            $this->_courses[$course->id] = $course;
        }
    }

    /**
     * load all groups for current user
     */
    protected function _load_groups(){
        $groups = groups_get_my_groups();
        foreach ($groups as $group){
            if ($this->_groups_by_courseid[$group->courseid] ?? false){
                continue;
            }
            $this->_groups_by_courseid[$group->courseid] = $group;
        }
    }

    /**
     * load all available secondary libs
     */
    protected function _load_libs(){
        foreach (static::OTHER_PLUGINS as $key_name => $plugin){
            if (NED::is_plugin_exists($plugin)){
                $key = '_has_' . $key_name;
                $this->$key = true;
            }
        }
    }

    /**
     * Load current page type
     */
    protected function _load_pagetypes(){
        $this->_pagetype = static::PAGETYPE_PEOPLE;
        if ($this->_capability > static::CAP_SEE_OWN_DASHBOARD){
            $pagetype = optional_param(static::P_PAGETYPE, static::PAGETYPE_PEOPLE, PARAM_TEXT);
            if (in_array($this->_pagetype, static::PAGETYPES)){
                $this->_pagetype = $pagetype;
            }
        }
    }

    /**
     * @return string
     */
    public function get_chosen_type(){
        return static::$_course_types[$this->_course_type] ?? '';
    }

    /**
     * Get course progress and set plot to show it
     *
     * @param        $progress
     * @param string $parent_selector
     *
     * @return \void
     */
    public function plot_completed($progress, $parent_selector=''){
        $plot = NED::$plotly::new_plotly("$parent_selector .course-complete .plot");
        $plot->add_value($progress, '', static::COLOR_GREEN);
        $plot->set_donut_piece('', 100,0);
        $plot->get_min = true;
    }

    /**
     * Set plot to show kica average course grade
     * Return kica grades
     *
     * @param array $kica_grades
     * @param string $parent_selector
     *
     * @return array['stat' => string, 'text' => string, 'title' => string];
     */
    public function plot_kica_items($kica_grades, $parent_selector=''){
        if (!$this->_has_kica || !$kica_grades){
            return null;
        }

        $kica_items = [];
        foreach ($kica_grades as $key => $grade){
            $color = static::KICA_COLORS[$key] ?? 'yellow';
            $value = $grade['value'] ?? 0;
            $max = $grade['max'] ?? 100;
            $title = get_string($key, 'local_kica');
            $text = $title[0] ?? '';

            $plot = NED::$plotly::new_plotly("$parent_selector .course-kica .$key .plot");
            $plot->set_bar(false, $max);
            $plot->add_value($value, $text, $color);
            $plot->hovertemplate = '';

            $kica_items[] = ['stat' => $key, 'text' => $text, 'title' => $title];
        }

        return $kica_items;
    }

    /**
     * Get course grade and set plot to show it
     *  Return grade
     *
     * @param null|int|float $grade
     * @param string         $parent_selector
     *
     * @return \void
     */
    public function plot_course_grade($grade, $parent_selector=''){
        $s_grade = is_null($grade) ? '-' : (round($grade) . '%');

        $plot = NED::$plotly::new_plotly("$parent_selector .course-grade .plot");
        $plot->add_value($grade, '', static::COLOR_BLUE);
        $plot->set_donut_piece($s_grade, 100, 0.8);
        $plot->get_min = true;
        $plot->annotation_size = 15;
    }

    /**
     * Return "show inactive courses" checkbox
     *
     * @return string
     */
    public function get_show_inactive_checkbox(){
        return NED::single_checkbox2($this->get_url(), $this->_show_inactive, static::P_SHOWINACTIVE,
            NED::fa('fa-eye'), NED::fa('fa-eye-slash'), 'show-inactive-checkbox',
            NED::str('hideinactivecourses'), NED::str('showinactivecourses'));
    }

    /**
     * Return url to the dashboard page
     *
     * @return \moodle_url
     */
    public function get_url(){
        $params = [];
        if ($this->_chosen_schoolid && count($this->_schools) > 1){
            $params[static::P_SCHOOLID] = $this->_chosen_schoolid;
        }
        if ($this->_chosen_userid && count($this->_school_users) > 1){
            $params[static::P_USERID] = $this->_chosen_userid;
        }
        if ($this->_show_inactive){
            $params[static::P_SHOWINACTIVE] = $this->_show_inactive;
        }
        if ($this->_pagetype != static::PAGETYPE_PEOPLE){
            $params[static::P_PAGETYPE] = $this->_pagetype;
        }
        return new \moodle_url(static::PAGE_DASHBOARD, $params);
    }

    /**
     * Section control form for render method
     *
     * @return string
     */
    public function get_control_form(){
        if (empty($this->_schools) || $this->capability <= static::CAP_SEE_OWN_DASHBOARD){
            return '';
        }

        $form = [];
        $url = $this->get_url();

        // choose school
        $school_opts = [];
        $attr = [];
        $count = count($this->_schools);
        if ($count != 1){
            $school_opts[0] = NED::str('allschools');
        }
        if ($count <= 1){
            $attr['disabled'] = 'disabled';
        }
        foreach ($this->_schools as $school){
            $school_opts[$school->id] = $school->name;
        }
        $form[] = NED::single_autocomplete($url, static::P_SCHOOLID, $school_opts, $this->chosen_schoolid,
            NED::fa('fa-university'), '', $attr);
        // choose user
        $attr = [];
        $user_opts = [];
        $count = count($this->_school_users);
        if ($count != 1){
            $user_opts[0] = NED::str('allusers');
        }
        if ($count <= 1){
            $attr['disabled'] = 'disabled';
        }
        $user_opts += $this->_school_users;
        $form[] = NED::single_autocomplete($url, static::P_USERID, $user_opts, $this->chosen_userid,
            NED::fa('fa-user'), '', $attr);
        //
        $this->c->control_form = join('', $form);
        return $this->c->control_form;
    }

    /**
     * Section control pagetype form for render method
     *
     * @return string
     */
    public function get_pagetype_change_form(){
        if ($this->_capability <= static::CAP_SEE_OWN_DASHBOARD){
            return '';
        }

        $form = [];
        $base_url = $this->get_url();
        $base_classes = ['btn'];

        foreach (static::PAGETYPES as $pagetype){
            $url = clone($base_url);
            $classes = $base_classes;
            $classes[] = ($this->_pagetype == $pagetype) ? 'btn-primary' : 'btn-secondary';
            $url->param(static::P_PAGETYPE, $pagetype);
            $form[] = NED::link($url, NED::str('pagetype_'.$pagetype), $classes);
        }

        if (NED::is_ai_exists()) {
            $form[] = NED::link(
                new \moodle_url('/local/academic_integrity/infractions.php'),
                NED::str('pluginname', null, 'local_academic_integrity'), 'btn btn-primary');
        }

        $this->c->pagetype_change_form = join('', $form);
        return $this->c->pagetype_change_form;
    }

    /**
     * @return array|null
     */
    public function get_dn_content(){
        if ($this->_capability <= static::CAP_SEE_OWN_DASHBOARD){
            return null;
        }

        $userids = $this->_chosen_userid;
        $groupids = null;
        $schoolids = null;

        if ($this->has_schoolmanager){
            if ($this->_chosen_schoolid){
                $schoolids = $this->_chosen_schoolid;
            } elseif ($this->_capability < static::CAP_SEE_ALL_DASHBOARDS){
                $schoolids = array_keys($this->_schools);
                if (empty($schoolids)){
                    return null;
                }
            }

            if ($this->_capability < static::CAP_SEE_OWN_COHORT_DASHBOARDS){
                $userids = array_keys($this->_school_users);
                if (empty($userids)){
                    return null;
                }
                $groupids = NED::get_user_groupids(0, $this->_viewerid);
                if (empty($groupids)){
                    return null;
                }
            }
        } else {
            if ($this->_capability < static::CAP_SEE_ALL_DASHBOARDS) {
                $userids = array_keys($this->_school_users);
                if (empty($userids)){
                    return null;
                }
            }
        }

        $this->c->dn_data = DN::get_table_summary($userids, $groupids, $schoolids, $this->get_url());
        return $this->c->dn_data;
    }

    /**
     * Section user data for render method
     *
     * @return void
     */
    public function get_user_info(){
        $user = $this->user;
        $this->c->userpicture = $this->o->user_picture($user, ['size' => 100, 'link' => true, 'class' => 'frontpage-userpicture']);
        $this->c->username = fullname($user);
        $this->c->userlink = new \moodle_url('/user/profile.php', ['id' => $this->userid]);
        $this->c->academic_integrity = NED::get_ai_flag_status($user);
        $this->c->academic_integrity_flag = NED::get_ai_flag($this->userid);

        // location
        $location = [];
        if ($user->city ?? false){
            $location[] = $user->city;
        }
        if ($user->country ?? false){
            $location[] = get_string($user->country, 'countries');
        }
        $this->c->location = join(', ', $location);

        // lastlogin
        if ($user->lastaccess ?? false) {
            $t = time() - $user->lastaccess;
            if ($t > 0){
                $lastlogin = get_string('ago', 'message', format_time($t));
            } else {
                $lastlogin = get_string('now');
            }
        } else {
            $lastlogin = get_string('never');
        }
        $this->c->lastlogin = $lastlogin;
        // logo and school_year
        if ($this->has_schoolmanager && ($this->school->logo ?? false)){
            if ($logo_url = LSM\school_manager::get_logo_url($this->school->id)){
                $this->c->school_logo = \html_writer::img($logo_url, 'logo');
            }
        }
        $this->c->school_year_start = $this->_school->startdate ?? 0;
        $this->c->school_year_end = $this->_school->enddate ?? 0;
        // school
        $this->c->school_name = $this->school->name ?? '';
        $this->c->academic_program = $user->profile[static::FIELD_ED_PROGRAM] ?? '';
        $this->c->expected_graduation = $user->profile[static::FIELD_GRADUATION] ?? '';
    }

    /**
     * Section data for render method
     *
     * @return \stdClass
     */
    public function get_summary_section(){
        $summary_section = new \stdClass();
        $get_array_percent = function($list){
            if (empty($list)){
                return 0;
            }
            $val = array_sum($list)/count($list);
            return round($val);
        };

        $course_count = ['completed' => 0, 'inprogress' => 0];
        $grade_average = ['completed' => [], 'inprogress' => [], 'overall' => []];
        $kica_grades = [];
        $kica_empty = true;

        if ($this->_has_kica){
            foreach (NED::KICA_KEYS as $key){
                $kica_grades[$key] = [];
            }
        }
        $participation_power = ['completed' => [], 'inprogress' => [], 'overall' => []];
        $userid = $this->userid;
        foreach ($this->_courses as $course){
            if ($course->course_type != static::COURSES_FOR_GRAPHS){
                continue;
            }
            $progress = $course->get_progress($userid);
            $completed = $progress == 100;
            $st = $completed ? 'completed' : 'inprogress';
            $course_count[$st]++;

            $grade = $course->get_course_grade($userid);
            $grade_average[$st][] = $grade;
            $grade_average['overall'][] = $grade;

            if ($this->_has_kica && $course->kica_enabled){
                $kg = $course->get_kica_average($userid);
                foreach (NED::KICA_KEYS as $key){
                    $param = $kg[$key];
                    if (empty($param['percent'])){
                        $result = $param['value'];
                    } else {
                        $result = $param['max'] ? ($param['value'] / $param['max'] * 100) : 0;
                    }
                    $kica_grades[$key][] = $result;
                }
                $kica_empty = false;
            }

            if ($course->is_pp_on()){
                $pp = $course->get_participation_power($userid);
                $participation_power[$st][] = $pp;
                $participation_power['overall'][] = $pp;
            }
        }

        // Courses
        $courses = new \stdClass();
        $plot = NED::$plotly::new_plotly('.summary .courses .plot');
        $courses->legend = [
            'completed' => ['color' => static::COLOR_GREEN],
            'inprogress' =>['color' => static::COLOR_BLUE],
        ];

        $sum = 0;
        foreach ($courses->legend as $key => &$item){
            $item['name'] = $key;
            $item['value'] = $course_count[$key] ?? 0;
            $sum += $item['value'];
            $item['text'] = NED::str($key);
            $plot->add_value($item['value'], $item['text'], $item['color']);
        }

        if ($sum > 0){
            $plot->set_donut('', 0.7);
            $plot->hoverinfo = 'label+percent';
            $plot->textinfo = 'text';
        } else {
            $plot->set_donut_piece('', 100, 0.7);
        }
        $courses->legend = array_values($courses->legend);
        $summary_section->courses = [$courses];

        // Grade average
        $ga = new \stdClass();
        $ga->stats = [
            'completed'  => ['color' => static::COLOR_GREEN],
            'inprogress' => ['color' => static::COLOR_BLUE],
            'overall'    => [],
        ];

        $all_grades = count($grade_average['overall']);
        $overall_plot = NED::$plotly::new_plotly(".summary .grade-average .stat.overall .plot");
        foreach ($ga->stats as $key => &$stat){
            $stat['stat'] = $key;
            $stat['text'] = NED::str($key);
            $grade_stat = $grade_average[$key] ?? [];
            $stat['value'] = $val = $get_array_percent($grade_stat);
            if ($key != 'overall'){
                $plot = NED::$plotly::new_plotly(".summary .grade-average .stat.$key .plot");
                $plot->add_value($stat['value'], '', $stat['color']);
                $v = $all_grades ? round($val/$all_grades * count($grade_stat)) : 0;
                $overall_plot->add_value($v, '', $stat['color']);
            } else {
                $plot = &$overall_plot;
            }

            if (!empty($stat['value'] )) {
                $plot->set_donut_piece($stat['value'] . '%', 100, 0.8);
            } else {
                $plot->set_donut_piece('', 100, 0.8);
            }

            $plot->annotation_size = 15;
        }
        $ga->stats = array_values($ga->stats);
        $summary_section->grade_average = $ga;

        // KICA Performance
        if ($this->_has_kica && !$kica_empty){
            $kp = new \stdClass();
            $kp->stats = [];

            foreach (static::KICA_COLORS as $key => $color){
                $value = $get_array_percent($kica_grades[$key] ?? []);
                $text = get_string($key, 'local_kica');

                $plot = NED::$plotly::new_plotly(".summary .kica-performance .$key .plot");
                $plot->set_bar();
                $plot->hovertemplate .= '%';
                $plot->add_value($value, $text, $color);
                $plot->hover_size = 10;

                $kp->stats[] = ['stat' => $key, 'text' => $text];
            }

            $summary_section->kica_performance = $kp;
        }

        // Participation Power
        if (!empty($participation_power['overall'])){
            $pp = new \stdClass();
            $pp->stats = [
                'completed'     => [],
                'inprogress'    => [],
                'overall'       => [],
            ];
            foreach ($pp->stats as $key => &$stat){
                $stat['stat'] = $key;
                $stat['text'] = NED::str($key);
                $stat['pp_value'] = $get_array_percent($participation_power[$key] ?? []);
                if (!empty($stat['pp_value'])) {
                    $stat['ishigh'] = true;
                    $stat['pp_status'] = course::get_participation_power_status_by_power($stat['pp_value']);
                    $stat['pp_text'] =  NED::str($stat['pp_status']);
                } else {
                    $stat['pp_status'] = course::get_participation_power_status_by_power($stat['pp_value']);
                    $stat['pp_text'] =  '';
                }
            }
            $pp->stats = array_values($pp->stats);
            $summary_section->participation_power = $pp;
        }

        $this->c->summary_section = $summary_section;

        return $summary_section;
    }

    /**
     * Section data for render method
     *
     * @return array of rendered courses for course_overview
     */
    public function get_course_overview(){
        $this->c->course_types = [];
        $url = $this->get_url();
        $userid = $this->userid;
        $chosen_course_type = $this->get_chosen_type();
        if ($chosen_course_type){
            foreach (static::$_course_types as $key => $course_type){
                if ($chosen_course_type == $course_type){
                    $add = \html_writer::span($course_type, 'active');
                } else {
                    $url->param(static::P_COURSETYPE, $key);
                    $add = \html_writer::link($url, $course_type);
                }
                $this->c->course_types[] = \html_writer::div($add, 'course-type');
            }
        }

        $co = [];
        foreach ($this->courses as $course){
            if ($course->course_type != $chosen_course_type){
                continue;
            }
            $courseid = $course->id;
            $c = new \stdClass();
            $c->id = $courseid;
            $c->courseurl = (new \moodle_url('/course/view.php', ['id' => $courseid]))->out();
            $c->name = $course->get_formatted_fullname();
            $c->startdate = $course->startdate;
            $c->enddate = $course->enddate;

            $groupid = $this->groups_by_courseid[$courseid]->id ?? 0;
            $c->onlineteachers = $course->get_role_user_names('online-teacher', $groupid);
            $c->classroomteachers = $course->get_role_user_names('classroom-teacher', $groupid);

            $parent_selector = ".course-overview .fd-course-card[data-courseid='$courseid']";

            $c->code = $course->code;
            $c->progress = $course->get_progress($userid);
            $this->plot_completed($c->progress, $parent_selector);
            $c->kica_items = $this->plot_kica_items($course->get_kica_average($userid), $parent_selector);
            if ($course->is_pp_on()){
                $c->participation_power = $course->get_participation_power_item($userid);
            }
            $this->plot_course_grade($course->get_course_grade($userid), $parent_selector);

            $co[] = $c;
        }

        $this->c->show_inactive = $this->get_show_inactive_checkbox();
        $this->c->course_overview = $co;

        return $this->c->course_overview;
    }

    /**
     * Section badge for render method
     *
     * @return mixed
     */
    public function get_badge_block(){
        /** @var \theme_ned_boost\output\core\badges_renderer $output */
        $output = $this->o->page->get_renderer('core', 'badges');
        $this->c->badge_block = $output->print_badges_list(null, $this->userid);

        return $this->c->badge_block;
    }

    /**
     * Section calendar for render method
     *
     * @return string
     */
    public function get_calendar_block(){
        global $USER;
        if (!$this->_userid){
            return '';
        }

        $USER = $this->_user;
        $data = $template = null;
        try {
            $calendar = \calendar_information::create(time(), SITEID);
            [$data, $template] = calendar_get_view($calendar, 'upcoming_mini', false);
        } catch (\Throwable $ex) {
            return '';
        } finally {
            $USER = $this->_viewer;
        }

        /** @var \core_calendar_renderer $renderer */
        $renderer = $this->o->page->get_renderer('core_calendar');
        $this->c->calendar_block = $renderer->render_from_template($template, $data);
        return $this->c->calendar_block;
    }

    /**
     * Render student list view section
     *
     * @return \stdClass
     */
    public function get_student_list_view(){
        $sch_id = $this->chosen_schoolid;
        $crew_names = [];
        $sm_users = [];

        if ($this->has_schoolmanager){
            $SM = LSM\school_manager::get_school_manager();
            $crew_names = $SM->get_crew_names($sch_id);
            $sm_users = $SM->get_school_students($sch_id);
        }

        $student_list = new \stdClass();
        if ($this->chosen_schoolid){
            $name = $this->_school->name ?? '';
            $code = $this->_school->code ?? ($this->_school->idnumber ?? false);
            if ($code){
                $name = trim(str_replace($code, '', $name), ' -');
                if (empty($name)){
                    $name = $this->_school->name ?? '';
                }
            }
            $student_list->school_name = $name;
            if ($this->_has_schoolmanager){
                $student_list->school_url = LSMR::get_url($this->_school->id);
            }
            $student_list->school_year_start = $this->_school->startdate ?? 0;
            $student_list->school_year_end = $this->_school->enddate ?? 0;
            $student_list->school_year = $student_list->school_year_start || $student_list->school_year_end;
            $location = [];
            if ($this->_school->city ?? false){
                $location[] = $this->_school->city;
            }
            if ($this->_school->country ?? false){
                $location[] = get_string($this->_school->country, 'countries');
            }
            $student_list->location = join(', ', $location);
        }

        $student_list->school_users = [];

        foreach ($this->_school_users as $uid => $username){
            $u = [
                'userid' => $uid,
                'name' => $username,
                'academic_integrity' => NED::get_ai_flag($uid)
            ];

            $url = $this->get_url();
            $url->param(static::P_USERID, $uid);
            $u['userlink'] = $url;
            if ($this->has_schoolmanager && isset($sm_users[$uid])){
                $u_sch_id = $sm_users[$uid]->schoolid ?? 0;
                if ($crewid = ($sm_users[$uid]->crewid ?? false)){
                    $u['cohortid'] = $crewid;
                    $u['cohortname'] = $sch_id ? ($crew_names[$crewid] ?? '') : ($crew_names[$u_sch_id][$crewid] ?? '');
                    $u['cohortlink'] = LSMR::get_url($u_sch_id, $crewid, LSMR::PAGE_USER);
                    $u['cohorttype'] = 'crew';
                }
            }

            $student_list->school_users[] = $u;
        }

        $this->c->student_list_view = $student_list;
        return $this->c->student_list_view;
    }

    /**
     * Get courses types from custom fields
     *  return them as array, or null, if them not exist
     *
     * @return array|null
     */
    static public function get_course_types(){
        global $DB;
        static $res = false;

        if ($res === false){
            $res = null;
            do {
                $course_types = $DB->get_record('customfield_field', ['shortname' => static::FIELD_COURSE_TYPE]);
                if (!$course_types){
                    break;
                }
                /** @var \customfield_select\field_controller $fc */
                $fc = \core_customfield\field_controller::create(0, $course_types);
                $options = array_slice($fc->get_options(), 1);
                if (empty($options)){
                    break;
                }

                $res = $options;
            } while(false);
        }
        return $res;
    }

    /**
     * Return capability to see own or other dashboards.
     *  All upper capability includes others, so you can check by using ">" or "<"
     *
     * @param null $ctx - context
     *
     * @return int
     */
    static public function get_dashboard_capability($ctx=null){
        $ctx = $ctx ?? \context_system::instance();
        if (has_capability('theme/ned_boost:seealldashboards', $ctx)){
            return static::CAP_SEE_ALL_DASHBOARDS;
        }
        if (has_capability('theme/ned_boost:seeowncohortdashboards', $ctx)){
            return static::CAP_SEE_OWN_COHORT_DASHBOARDS;
        }
        if (has_capability('theme/ned_boost:seeowngroupsdashboards', $ctx)){
            return static::CAP_SEE_OWN_GROUPS_DASHBOARDS;
        }
        if (has_capability('theme/ned_boost:seeowndashboard', $ctx)){
            return static::CAP_SEE_OWN_DASHBOARD;
        }
        return static::CAP_CANT_SEE_DASHBOARD;
    }

    /**
     * @param array $cohorts - should be [cohortid => cohort]
     *
     * @return array
     */
    static public function get_schools_by_cohorts($cohorts=[]){
        global $DB;
        if (NED::is_schm_exists() && !empty($cohorts)){
            [$sql_cohorts, $params] = $DB->get_in_or_equal(array_keys($cohorts), SQL_PARAMS_NAMED);
            return $DB->get_records_select(LSM\school_manager::TABLE_SCHOOL, 'id '.$sql_cohorts, $params, 'name');
        }

        $schools = [];
        foreach ($cohorts as $id => $cohort){
            $code = $cohort->idnumber ?? '';
            // Schools have 4-digits code
            if (strlen($code) == 4){
                $schools[$id] = $cohort;
            }
        }
        uasort($schools, function($a, $b){ return $a->name <=> $b->name;});
        return $schools;
    }

    /**
     * Return all members of cohorts with id in $school_ids
     *
     * @param array $school_ids
     * @param array $role_names
     *
     * @return array [userid => user_fullname]
     */
    static public function get_school_members($school_ids=[], $role_names=[]){
        global $DB;
        if (empty($school_ids)){
            return [];
        }

        $school_ids = NED::val2arr($school_ids);
        $role_names = NED::val2arr($role_names);
        $params = [];
        $select = [
            "SELECT DISTINCT u.id, CONCAT(u.firstname, ' ', u.lastname) 
             FROM {user} u 
             JOIN {cohort_members} cm
                ON u.id = cm.userid"
        ];
        $where = [];
        [$school_where, $school_params] = $DB->get_in_or_equal($school_ids, SQL_PARAMS_NAMED, 'cohortid');
        $where[] = 'cm.cohortid ' . $school_where;
        $params = array_merge($params, $school_params);

        if (static::$_default_role_field_id && !empty($role_names)){
            $field_id = static::$_default_role_field_id;
            $select[] = "
                LEFT JOIN {user_info_data} uid
                    ON uid.userid = u.id AND uid.fieldid = '$field_id'
            ";

            [$where_role, $rn_params] = $DB->get_in_or_equal($role_names, SQL_PARAMS_NAMED, 'rolename');
            $params = array_merge($params, $rn_params);

            if (in_array(static::$_default_role_default_value, $role_names)){
                $where_role .= ' OR uid.data IS NULL';
            }

            $where[] = 'uid.data ' . $where_role;
        }

        $select = join('', $select);
        $where = empty($where) ? '' : ('WHERE (' . join(') AND (', $where) . ')');
        $members = $DB->get_records_sql_menu("$select $where ORDER BY u.id", $params);
        asort($members);
        return $members;
    }

    /**
     * Function to export the renderer data in a format that is suitable for a
     * mustache template. This means:
     * 1. No complex types - only \stdClass, array, int, string, float, bool
     * 2. Any additional info that is required for the template is pre-calculated (e.g. capability checks).
     *
     * @param \renderer_base $output Used to do a final render of any components that need to be rendered for export.
     *
     * @return \stdClass|array
     */
    public function export_for_template(\renderer_base $output){
        global $SITE;
        $this->c->output = $this->o;

        if ($this->_capability < static::CAP_SEE_OWN_DASHBOARD){
            $this->c->show = false;
            return $this->c->export();
        }
        $this->c->show = true;
        $this->c->has_schoolmanager = $this->_has_schoolmanager;

        if (empty($this->o->page->layout_options['nonavbar'])) {
            $this->c->shownavdrawer = true;
            user_preference_allow_ajax_update('drawer-open-nav', PARAM_ALPHA);
        }
        $navigation = $this->o->get_frontdashboard_context();

        $this->c->regionmainsettingsmenu = $navigation['regionmainsettingsmenu'];
        $this->c->hasregionmainsettingsmenu = $navigation['hasregionmainsettingsmenu'];
        $this->c->primarymoremenu = $navigation['primarymoremenu'];
        $this->c->secondarymoremenu = $navigation['secondarymoremenu'];
        $this->c->mobileprimarynav = $navigation['mobileprimarynav'];
        $this->c->langmenu = $navigation['langmenu'];
        $this->c->usermenu = $navigation['usermenu'];
        $this->c->overflow = $navigation['overflow'];
        $this->c->bodyattributes = $navigation['bodyattributes'];

        $this->c->sitename = format_string($SITE->shortname, true,
            ['context' => \context_course::instance(SITEID), "escape" => false]);
        $this->c->this_url = $this->get_url()->out(false);

        if ($this->_capability == static::CAP_SEE_OWN_DASHBOARD){
            $this->c->userid = $this->userid;
        } else {
            $this->c->userid = $this->_chosen_userid;
            $this->get_control_form();
            // $this->get_pagetype_change_form();
        }

        if ($this->_userid && ($this->_viewerid != $this->_userid)){
            $this->c->add_classes[] = 'view-other-user';
        }

        if ($this->_capability == static::CAP_SEE_OWN_DASHBOARD || $this->c->userid || !$this->has_schoolmanager){
            $this->get_user_info();
        }

        $pagetype = 'pagetype_'.$this->_pagetype;
        $this->c->$pagetype = true;

        if ($this->_pagetype == static::PAGETYPE_PEOPLE){
            if ($this->_capability == static::CAP_SEE_OWN_DASHBOARD || $this->c->userid || !$this->has_schoolmanager){
                $this->get_summary_section();
                $this->get_course_overview();
                $this->get_badge_block();
                $this->get_calendar_block();
            } else {
                $this->get_student_list_view();
            }

            $this->c->secondary_column = !empty($this->c->badge_block);
            $this->c->plotly_add_elements = NED::$plotly::render_all(true, false, true);

        } elseif ($this->_pagetype == static::PAGETYPE_DN){
            $this->get_dn_content();
            $this->c->dn_data_exist = !empty($this->c->dn_data);
            if ($this->c->dn_data_exist){
                NED::$C::js_call_amd('add_sorter', 'add_sort', '.nedtable');
            }
        }

        $this->c->school_year = $this->c->school_year_start || $this->c->school_year_end;
        $this->c->bodyattributes = $this->o->body_attributes($this->c->add_classes);

        return $this->c->export();
    }

    /**
     * Render this class element
     * @param bool $return
     *
     * @return string
     */
    public function render($return=false){
        $res = $this->o->render($this);
        if (!$return){
            echo $res;
        }
        return $res;
    }
}

/**
 * Class dashboard_content
 *
 * @package theme_ned_boost\output
 */
class dashboard_content extends \stdClass{
    use \local_ned_controller\base_empty_class;
    public $show = false;
    public $dn_data_exist = false;
    public $bodyattributes;
    public $add_classes = [];
    public $output;
    //new menu parameters
    public $hasregionmainsettingsmenu;
    public $primarymoremenu;
    public $secondarymoremenu;
    public $mobileprimarynav;
    public $usermenu;
    public $langmenu;
    public $overflow;

    public $sitename;
    public $this_url = '#';
    public $shownavdrawer = false;
    public $show_inactive = '';
    public $course_types = [];
    public $control_form = '';
    public $pagetype_change_form = '';
    public $student_list_view;
    public $userid = 0;

    public $userpicture = '';
    public $username = '';
    public $userlink = '';
    public $location = '';
    public $lastlogin = '';
    public $school_name = '';
    public $school_logo = '';
    public $school_year = false;
    public $school_year_start = 0;
    public $school_year_end = 0;
    public $academic_program = '';
    public $admission_date = 0;
    public $expected_graduation = 0;
    public $academic_integrity = '';

    public $summary_section;
    public $course_overview;
    public $badge_block;
    public $calendar_block;
    public $dn_data;

    public $secondary_column;

    public $plotly_add_elements = '';


    /**
     * dashboard_content constructor.
     *
     * @param null $var
     */
    public function __construct($var=null){
        if (!is_null($var)){
            $this->import($var);
        }
    }

    /**
     * @param string $name
     * @param mixed  $value
     *
     * @return mixed
     */
    public function __set($name, $value){
        $this->$name = $value;
        return $value;
    }

    /**
     * Import $object data into $this
     *
     * @param $object
     */
    public function import($object){
        $this->copy($object, '', false);
    }
}

/**
 * @property-read \context $ctx;
 * @property-read array $course_fields;
 * @property-read int $badge_field_id;
 * @property-read int $course_type_field_id;
 * @property-read bool $site_has_sm;
 * @property-read bool $site_has_kica;
 * @property-read \stdClass $kica = null;
 * @property-read bool $kica_enabled = false;
 * @property-read \grade_item $grade_item;
 * @property-read string $code = null;
 * @property-read string $course_type = '';
 * @property-read \stdClass $tt_config
 *
 * @property-read \stdClass $record;
 * @property-read \stdClass $data;
 */
class course extends \core_course_list_element {
    const FIELD_CODE = 'badge_code';
    const FIELD_COURSE_TYPE = 'course_type';

    const OTHER_PLUGINS = [
        'kica' => NED::KICA,
        'sm' => NED::SM,
        'tt' => NED::TT,
    ];

    const PP_HIGH = 'high';
    const PP_MEDIUM = 'medium';
    const PP_LOW = 'low';

    const GET_PROPERTIES = [
        'grade_item' => 'get_grade_item',
        'kica' => 'get_kica',
        'kica_enabled' => 'get_kica_enabled',
        'code' => 'get_code',
        'course_type' => 'get_course_type',
        'tt_config' => 'get_tt_config',
    ];

    static protected $_course_fields = null;
    static protected $_badge_field_id = null;
    static protected $_course_type_field_id = null;

    static protected $_site_has_sm = false;
    static protected $_site_has_kica = false;
    static protected $_site_has_tt = false;

    static protected $_site_config_tt = null;

    /** @var array $_properties_method_data - for _check_get_property method only */
    protected $_properties_method_data = [];
    /** @var array $_user_method_data - for _check_user_functions method only */
    protected $_user_method_data = [];

    /** @var \context $_ctx */
    protected $_ctx;
    protected $_kica = null;
    protected $_kica_enabled = false;
    /** @var \grade_item $_grade_item */
    protected $_grade_item = null;
    protected $_progress = 0;
    protected $_data;
    protected $_code = null;
    protected $_course_type = '';

    /**
     * Creates an instance of the class from record
     *
     * @param \stdClass $record except fields from course table it may contain
     *                         field hassummary indicating that summary field is not empty.
     *                         Also it is recommended to have context fields here ready for
     *                         context preloading
     */
    public function __construct(\stdClass $record){
        parent::__construct($record);

        static::_load_libs();
        static::_load_static_data();

        $this->_ctx = \context_course::instance($this->id);
        $this->_data = new \stdClass();
    }

    /**
     * load all available secondary libs
     */
    static protected function _load_libs(){
        foreach (static::OTHER_PLUGINS as $key_name => $plugin){
            if (NED::is_plugin_exists($plugin)){
                $key = '_site_has_' . $key_name;
                static::$$key = true;
            }
        }

        if (static::$_site_has_tt){
            static::$_site_config_tt = get_config(TT\PLUGIN_NAME);
        }
    }

    /**
     * load static class data
     */
    static protected function _load_static_data(){
        static $done = false;

        if ($done){
            return;
        }

        global $DB;

        if (is_null(static::$_course_fields)){
            static::$_course_fields = [];
            $columns = $DB->get_columns('course');
            foreach ($columns as $column){
                $name = $column->name;
                static::$_course_fields[$name] = $name;
            }
        }
        if (is_null(static::$_badge_field_id)){
            static::$_badge_field_id = $DB->get_field('customfield_field', 'id', ['shortname' => static::FIELD_CODE]);
        }
        if (is_null(static::$_course_type_field_id)){
            static::$_course_type_field_id = $DB->get_field('customfield_field', 'id', ['shortname' => static::FIELD_COURSE_TYPE]);
        }

        $done = true;
    }

    /**
     * Retrieve fields summary and summaryformat together because they are most likely to be used together.
     */
    protected function _get_summary(){
        global $DB;
        $record = $DB->get_record('course', ['id' => $this->record->id], 'summary, summaryformat');
        $this->record->summary = $record->summary;
        $this->record->summaryformat = $record->summaryformat;
    }

    /**
     * Return field from table 'course' that was not retrieved.
     *
     * @param $name
     *
     * @return mixed
     */
    protected function _get_field_from_db($name){
        global $DB;
        $this->record->$name = $DB->get_field('course', $name, ['id' => $this->record->id]);
        return $this->record->$name;
    }

    /**
     * Magic method to check if property is set
     *
     * @param string $name
     * @return bool
     */
    public function __isset($name) {
        return isset($this->record->$name) || isset($this->_data->$name);
    }

    /**
     * Magic method to get a course, class or this property
     *
     * Returns any field from table course (retrieves it from DB if it was not retrieved before)
     *
     * @param string $name
     * @return mixed
     */
    public function __get($name) {
        $pr_name = '_' . $name;
        $res = null;
        if (static::GET_PROPERTIES[$name] ?? false){
            $method = static::GET_PROPERTIES[$name];
            return $this->$method();
        } elseif (property_exists($this->record, $name)) {
            return $this->record->$name;
        } elseif (property_exists($this, $pr_name)){
            $res = ($this::${$pr_name} ?? $this->$pr_name) ?? null;
        } elseif(property_exists($this, $name)){
            $res = ($this::${$name} ?? $this->$name) ?? null;
        } elseif ($name === 'summary' || $name === 'summaryformat') {
            $this->_get_summary();
            return $this->record->$name;
        } elseif (static::$_course_fields[$name] ?? false) {
            return $this->_get_field_from_db($name);
        } else {
            $res = $this->_data->$name ?? null;
        }

        if (is_object($res)){
            $res = clone($res);
        }

        return $res;
    }

    /**
     * All properties are read only, sorry.
     *
     * @param string $name
     */
    public function __unset($name) {
        if (isset($this->_data->$name)){
            unset($this->_data->$name);
        } else {
            debugging('Can not unset '.get_class($this).' instance properties!');
        }
    }

    /**
     * Magic setter method, we do not want anybody to modify properties from the outside
     *
     * @param string $name
     * @param mixed $value
     */
    public function __set($name, $value) {
        if (isset($this->record->$name) || isset(static::$_course_fields[$name]) || ($name === 'summary' || $name === 'summaryformat')){
            debugging('Can not change '.get_class($this).' instance properties!');
        } else {
            $this->_data->$name = $value;
        }
    }

    /**
     * Returns custom fields data for this course
     *
     * @return \core_customfield\data_controller[]
     */
    public function get_custom_fields() : array {
        if (!isset($this->record->customfields)) {
            $this->record->customfields = \core_course\customfield\course_handler::create()->get_instance_data($this->id, true);
        }
        return $this->record->customfields;
    }

    /**
     * Export $this data as \stdClass object
     *
     * @param bool $only_course_record
     * @return \stdClass
     */
    public function export($only_course_record=false){
        $res = new \stdClass();
        if (!$only_course_record){
            foreach ($this->_data as $key => $item){
                $res->$key = $item;
            }
        }
        foreach ($this->record as $key => $item){
            $res->$key = $item;
        }

        return $res;
    }

    /**
     * Return users fullnames by role shortname
     *
     * @param      $roleshortname
     * @param      $groupid
     * @param bool $linkable - return link to the users if true
     * @param bool $as_string - return string if true
     * @param bool $hide_multiple - dont' show results more then one if true
     *
     * @return array|string
     */
    public function get_role_user_names($roleshortname, $groupid, $linkable=false, $as_string=true, $hide_multiple=true){
        $users = static::get_role_users($roleshortname, $this->ctx, $groupid);
        $len = $users ? count($users) : 0;
        if ($as_string){
            if ($len < 1){
                return '';
            } elseif ($hide_multiple && $len > 1){
                return \html_writer::tag('i', NED::str('multiple'));
            }
        } elseif ($hide_multiple && $len > 1){
            $users = [reset($users)];
        }

        $users_names =  static::get_users_fullnames($users, $linkable, $this->id);
        if ($as_string){
            return join(', ', $users_names);
        }
        return $users_names;
    }

    /**
     * @param $roleshortname
     * @param $context
     * @param $groupid
     * @return array|bool
     */
    static public function get_role_users($roleshortname, $context, $groupid) {
        global $DB;
        static $role_ids = [];

        if (!isset($role_ids[$roleshortname])){
            $role = $DB->get_record('role', ['shortname' => $roleshortname]);
            $role_ids[$roleshortname] = $role ? $role->id : false;
        }

        $roleid = $role_ids[$roleshortname];
        if ($roleid) {
            if ($roleusers = get_role_users($roleid, $context, false, '', null, false, $groupid)) {
                return $roleusers;
            }
        }
        $role_ids = [];
        return [];
    }

    /**
     * @param \stdClass[] $users
     * @param bool        $linkable
     * @param null        $courseid
     *
     * @return array
     */
    static public function get_users_fullnames($users, $linkable=false, $courseid=null){
        $userfullnames = [];
        if (empty($users)){
            return $userfullnames;
        }

        foreach ($users as $user) {
            if ($linkable) {
                $params = ['id' => $user->id];
                if ($courseid){
                    $params['course'] = $courseid;
                }
                $url = new \moodle_url('/user/view.php', $params);
                $userfullnames[$user->id] = \html_writer::link($url, fullname($user));
            } else {
                $userfullnames[$user->id] = fullname($user);
            }
        }

        return $userfullnames;
    }

    /**
     * Get participation power status by participation power value
     *
     * @param $pp_value
     *
     * @return null|string
     */
    static public function get_participation_power_status_by_power($pp_value){
        if (!static::$_site_has_sm){
            return null;
        }

        return SM\activity_status::calc_high_low_result($pp_value);
    }

    //region User Functions
    /**
     * Check User Functions for saved data
     *
     * @param $function_name
     * @param $userid
     *
     * @return mixed
     */
    protected function _check_user_functions($function_name, $userid){
        if (!array_key_exists($userid, $this->_user_method_data[$function_name] ?? [])){
            $this->_user_method_data[$function_name][$userid] = $this->$function_name($userid, true);
        }
        return $this->_user_method_data[$function_name][$userid];
    }

    /**
     * Get the course progress percentage.
     *
     * @param      $userid
     * @param bool $force_recalculate = false
     *
     * @return int
     */
    public function get_progress($userid, $force_recalculate=false){
        if (!$force_recalculate){
            return $this->_check_user_functions(__FUNCTION__, $userid);
        }

        $progress = \core_completion\progress::get_course_progress_percentage($this->record, $userid);
        $progress = $progress ? round($progress) : 0;

        return $progress;
    }

    /**
     * Return participation power value
     *
     * @param      $userid
     * @param bool $force_recalculate = false
     *
     * @return null|int|float
     */
    public function get_participation_power($userid, $force_recalculate=false){
        if (!$force_recalculate){
            return $this->_check_user_functions(__FUNCTION__, $userid);
        }

        if (!static::$_site_has_sm){
            return null;
        }

        $user_activity = new SM\activity_status();
        $user_activity->check_activities_by_course($this->record, $userid);
        $participationpower = $user_activity->get_participation_power();
        return $participationpower;
    }

    /**
     * Return participation power status
     *
     * @param      $userid
     * @param bool $force_recalculate = false
     *
     * @return string
     */
    public function get_participation_power_status($userid, $force_recalculate=false){
        if (!$force_recalculate){
            return $this->_check_user_functions(__FUNCTION__, $userid);
        }

        $participationpower = $this->get_participation_power($userid);
        $pp_status = static::get_participation_power_status_by_power($participationpower);
        return $pp_status;
    }

    /**
     * Return participation power item
     *
     * @param      $userid
     * @param bool $force_recalculate = false
     *
     * @return array['pp_value' => int|float, 'pp_status' => string, 'pp_text' => string]
     */
    public function get_participation_power_item($userid, $force_recalculate=false){
        if (!$force_recalculate){
            return $this->_check_user_functions(__FUNCTION__, $userid);
        }

        $participationpower = $this->get_participation_power($userid);
        $pp_status = static::get_participation_power_status_by_power($participationpower);
        return ['pp_value' => $participationpower, 'pp_status' => $pp_status, 'pp_text' => NED::str($pp_status)];
    }

    /**
     * Return kica average grades by all kica categories for the course
     *
     * @param      $userid
     * @param bool $force_recalculate = false
     *
     * @return array|null ['value' => int, 'max' => int, 'percent' => bool]
     */
    public function get_kica_average($userid, $force_recalculate=false){
        if (!$force_recalculate){
            return $this->_check_user_functions(__FUNCTION__, $userid);
        }

        if (!$this->kica_enabled){
            return null;
        }

        $kica = $this->kica;
        $calc_natural = NED::is_kica_calculation_natural($kica);
        $kica_average = NED::kg_get_course_average($this->id, $userid);
        $grades = [];
        foreach (NED::KICA_KEYS as $kica_key){
            $grade = $kica_average[$kica_key] ?? null;
            $max = 100;
            $percent = true;
            if (!$calc_natural){
                $max = $kica->$kica_key ?? 0;
                $grade = $max ? ($grade/$max * 100) : 0;
                $percent = false;
            }
            $grades[$kica_key] = ['value' => round($grade), 'max' => $max, 'percent' => $percent];
        }

        return $grades;
    }

    /**
     * Return grade_grade for the user
     *
     * @param      $userid
     * @param bool $force_recalculate = false
     *
     * @return \grade_grade
     */
    public function get_grade_grade($userid, $force_recalculate=false){
        if(!$force_recalculate){
            return $this->_check_user_functions(__FUNCTION__, $userid);
        }

        $courseitem = $this->grade_item;
        $coursegrade = new \grade_grade(['itemid' => $courseitem->id, 'userid' => $userid]);
        $coursegrade->grade_item =& $courseitem;
        return $coursegrade;
    }

    /**
     * Return kica or moodle course grade
     *
     * @param      $userid
     * @param bool $force_recalculate = false
     *
     * @return int|float
     */
    public function get_course_grade($userid, $force_recalculate=false){
        if(!$force_recalculate){
            return $this->_check_user_functions(__FUNCTION__, $userid);
        }

        return NED::get_course_grade($this->id, $userid, 2, true, 0);
    }
        //endregion

    //region GET_PROPERTIES
    /*
     * GET_PROPERTIES section
     *  these functions calls ones and save result as static variable
     *  you can get all of them through magic method
     *
     * Warning: DON'T use required variables in these functions!
     */

    /**
     * Check GET_PROPERTIES function for saved data
     *
     * @param $function_name
     *
     * @return mixed
     */
    protected function _check_get_property($function_name){
        if (!array_key_exists($function_name, $this->_properties_method_data)){
            $this->_properties_method_data[$function_name] = $this->$function_name(true);
        }
        return $this->_properties_method_data[$function_name];
    }

    /**
     * Returns the grade item associated with the course
     *
     * @param $force_recalculate = false
     *
     * @return \grade_item
     */
    public function get_grade_item($force_recalculate=false){
        if (!$force_recalculate){
            return $this->_check_get_property(__FUNCTION__);
        }

        return \grade_item::fetch_course_item($this->id);
    }

    /**
     * Returns the kica item associated with the course
     *
     * @param $force_recalculate = false
     *
     * @return \stdClass
     */
    public function get_kica($force_recalculate=false){
        if (!$force_recalculate){
            return $this->_check_get_property(__FUNCTION__);
        }

        if (!static::$_site_has_kica){
            return null;
        }

        return NED::get_kica($this->id);
    }

    /**
     * Returns is kica enabled on this course
     *
     * @param $force_recalculate = false
     *
     * @return bool
     */
    public function get_kica_enabled($force_recalculate=false){
        if (!$force_recalculate){
            return $this->_check_get_property(__FUNCTION__);
        }

        $kica = $this->kica;
        if (!($kica->enabled ?? false)){
            return false;
        }

        return kica\helper::iskicaready($this->id);
    }

    /**
     * Return badge code for this course
     *
     * @param $force_recalculate = false
     *
     * @return string
     */
    public function get_code($force_recalculate=false){
        if (!$force_recalculate){
            return $this->_check_get_property(__FUNCTION__);
        }

        $code = '';
        do {
            if (!static::$_badge_field_id){
                break;
            }

            $customfields = $this->get_custom_fields();
            $field = $customfields[static::$_badge_field_id] ?? null;
            if (!$field){
                break;
            }

            $code = $field->export_value();
        } while(false);

        if (!$code){
            $name = $this->get_formatted_fullname();
            $code = $this->get_formatted_shortname();
            $code = strlen($code) < strlen($name) ? $code : '';
        }

        return $code;
    }

    /**
     * Return course type (form course_type field)
     * @param $force_recalculate = false
     *
     * @return string
     */
    public function get_course_type($force_recalculate=false){
        if (!$force_recalculate){
            return $this->_check_get_property(__FUNCTION__);
        }

        $course_type = '';
        if (!static::$_course_type_field_id){
            return $course_type;
        }

        $customfields = $this->get_custom_fields();
        $field = $customfields[static::$_course_type_field_id] ?? null;
        if (!$field){
            return $course_type;
        }

        $course_type = $field->export_value() ?: '';
        return $course_type;
    }

    /**
     * Return TT block config
     * @param $force_recalculate = false
     *
     * @return string
     */
    public function get_tt_config($force_recalculate=false){
        if (!$force_recalculate){
            return $this->_check_get_property(__FUNCTION__);
        }

        if (!static::$_site_has_tt){
            return null;
        }

        return TT\get_block_config($this->id, TT\PLUGIN);
    }
    //endregion

    /**
     * Does on this course participation power is turned on
     *
     * @return bool
     */
    public function is_pp_on(){
        $config = $this->tt_config;
        return $config->participationpower_option ?? false;
    }
}
