<?php
/** @noinspection PhpUnused */
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * @package    theme_ned_boost
 * @subpackage NED Boost
 * @copyright  NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @developer  G J Barnard - {@link http://about.me/gjbarnard} and
 *                           {@link http://moodle.org/user/profile.php?id=442195}
 */

defined('MOODLE_INTERNAL') || die();

use theme_ned_boost\shared_lib as NED;

include_once("$CFG->dirroot/user/classes/output/myprofile/manager.php");
include_once("$CFG->dirroot/user/profile/lib.php");
include_once("$CFG->dirroot/user/profile/definelib.php");
include_once("$CFG->dirroot/mod/url/locallib.php");
include_once("$CFG->dirroot/lib/weblib.php");
include_once("$CFG->dirroot/theme/ned_boost/local_lib.php");
include_once("$CFG->dirroot/local/ned_controller/lib.php");

/**
 * Inject SCSS content.
 *
 * Note: result of this function is saved in theme cache,
 *  so after changes of function work you need purging the caches or enabling theme designer mode
 *
 * @param theme_config $theme The theme config object.
 *
 * @return string
 */
function theme_ned_boost_get_main_scss_content($theme) {
    $toolbox = \theme_ned_boost\toolbox::get_instance();

    return $toolbox->get_main_scss_content($theme);
}

/**
 * Get SCSS to prepend.
 * Return the SCSS to prepend to main SCSS for this theme
 * @see theme_config::get_css_content_from_scss() - called from here
 *
 * Note: result of this function is saved in theme cache,
 *  so after changes of function work you need purging the caches or enabling theme designer mode
 *
 * @param theme_config $theme The theme config object.
 *
 * @return string The custom pre SCSS.
 */
function theme_ned_boost_get_pre_scss($theme) {
    $toolbox = \theme_ned_boost\toolbox::get_instance();

    return $toolbox->get_pre_scss($theme).local_ned_controller_get_pre_scss($theme);
}

/**
 * Inject additional SCSS.
 * Return the SCSS to append to our main SCSS for this theme
 * @see theme_config::get_css_content_from_scss() - called from here
 *
 * Note: result of this function is saved in theme cache,
 *  so after changes of function work you need purging the caches or enabling theme designer mode
 *
 * @param theme_config $theme The theme config object.
 *
 * @return string The custom post SCSS.
 */
function theme_ned_boost_get_extra_scss($theme) {
    $toolbox = \theme_ned_boost\toolbox::get_instance();

    return $toolbox->get_extra_scss($theme).local_ned_controller_get_extra_scss($theme);
}

/**
 * Parses CSS before it is cached.
 * This function can make alterations and replace patterns within the CSS.
 * @see theme_config::post_process() - called from here
 *
 * Note: result of this function is saved in theme cache,
 *  so after changes of function work you need purging the caches or enabling theme designer mode
 *
 * @param string $css The CSS
 * @param theme_config $theme The theme config object.
 *
 * @return string The parsed CSS The parsed CSS.
 */
function theme_ned_boost_process_css($css, $theme) {
    return local_ned_controller_process_css($css, $theme);
}

/**
 * Serves any files associated with the theme settings.
 *
 * @param stdClass $course.
 * @param stdClass $cm.
 * @param context $context.
 * @param string $filearea.
 * @param array $args.
 * @param bool $forcedownload.
 * @param array $options.
 * @return bool
 */
function theme_ned_boost_pluginfile($course, $cm, $context, $filearea, $args, $forcedownload, array $options=array()) {
    static $theme;
    if (empty($theme)) {
        $theme = theme_config::load(NED::$PLUGIN);
    }
    if ($context->contextlevel == CONTEXT_SYSTEM) {
        // By default, theme files must be cache-able by both browsers and proxies.  From 'More' theme.
        if (!array_key_exists('cacheability', $options)) {
            $options['cacheability'] = 'public';
        }
        if ($filearea === 'logo' || $filearea === 'loginlogo') {
            $theme = theme_config::load('ned_boost');
            // By default, theme files must be cache-able by both browsers and proxies.
            if (!array_key_exists('cacheability', $options)) {
                $options['cacheability'] = 'public';
            }
            return $theme->setting_file_serve($filearea, $args, $forcedownload, $options);
        } else if (preg_match("/^(institutionlogo|institutioncompactlogo)[1-9][0-9]*$/", $filearea)) {
            // Path hides the size, see: core_admin_pluginfile.
            global $CFG;

            $size = array_shift($args); // The path hides the size.
            $itemid = clean_param(array_shift($args), PARAM_INT);
            $filename = clean_param(array_shift($args), PARAM_FILE);
            $themerev = theme_get_revision();
            if ($themerev < 0) {
                // Normalise to 0 as -1 doesn't place well with paths.
                 $themerev = 0;
            }

            // Extract the requested width and height.
            $maxwidth = 0;
            /** @noinspection DuplicatedCode */
            $maxheight = 0;
            if (preg_match('/^\d+x\d+$/', $size)) {
                list($maxwidth, $maxheight) = explode('x', $size);
                $maxwidth = clean_param($maxwidth, PARAM_INT);
                $maxheight = clean_param($maxheight, PARAM_INT);
            }

            $lifetime = 0;
            if ($itemid > 0 && $themerev == $itemid) {
                // The itemid is $CFG->themerev, when 0 or less no caching. Also no caching when they don't match.
                $lifetime = DAYSECS * 60;
            }

            // Check if we've got a cached file to return. When lifetime is 0 then we don't want to cached one.
            $candidate = $CFG->localcachedir.'/'.NED::$PLUGIN_NAME."/$themerev/$filearea/{$maxwidth}x{$maxheight}/$filename";
            if (file_exists($candidate) && $lifetime > 0) {
                send_file($candidate, $filename, $lifetime, 0, false, false, '', false, $options);
            }

            // Find the original file.
            $fs = get_file_storage();
            $filepath = join('/',['', $context->id, NED::$PLUGIN_NAME, $filearea, '0', $filename]);
            /** @noinspection DuplicatedCode */
            if (!$file = $fs->get_file_by_hash(sha1($filepath))) {
                send_file_not_found();
            }

            // No need for resizing, but if the file should be cached we save it so we can serve it fast next time.
            if (empty($maxwidth) && empty($maxheight)) {
                if ($lifetime) {
                    file_safe_save_content($file->get_content(), $candidate);
                }
                send_stored_file($file, $lifetime, 0, false, $options);
            }

            // Proceed with the resizing.
            $filedata = $file->resize_image($maxwidth, $maxheight);
            if (!$filedata) {
                send_file_not_found();
            }

            // If we don't want to cached the file, serve now and quit.
            if (!$lifetime) {
                send_content_uncached($filedata, $filename);
            }

            // Save, serve and quit.
            file_safe_save_content($filedata, $candidate);
            send_file($candidate, $filename, $lifetime, 0, false, false, '', false, $options);
            return true;
        }
    }

    send_file_not_found();
    return false;
}

/**
 * @return bool
 */
function theme_ned_boost_should_add_showmyoverview_block(){
    global $PAGE;
    return $PAGE->pagelayout == 'frontpage'  && NED::get_config('showmyoverview') == 1;
}

/**
 * Let the theme initialise the page object (usually $PAGE).
 *
 * This may be used for example to request jQuery in add-ons.
 *
 * @param moodle_page $page
 */
function theme_ned_boost_page_init(moodle_page $page){
    global $DB;

    // change 'myoverview' block place
    //      block myoverview render in theme/ned_boost/layout/dynamiclayout_frontdashboard.php
    $add = theme_ned_boost_should_add_showmyoverview_block();
    $params = ['blockname' => 'myoverview', 'defaultregion' => 'side-post', 'defaultweight' => 3, 'showinsubcontexts' => 1,
        'pagetypepattern' => 'site-index'];
    $record = !empty($DB->get_record('block_instances', $params));
    if ($add and !$record){
        $page->blocks->add_block($params['blockname'], $params['defaultregion'],
            $params['defaultweight'], $params['showinsubcontexts'], $params['pagetypepattern']);
    } elseif (!$add and $record){
        $DB->delete_records('block_instances', $params);
    }

    // hide blocks from "my dashboard" page
    if ($page->pagelayout == 'mydashboard'){
        $page->blocks->show_only_fake_blocks(true);
    }
}

/**
 * @param string $option_name
 */
function theme_ned_boost_change_videofield($option_name=''){
    NED::change_videofield($option_name);
}

/**
 * @param \core_user\output\myprofile\tree $tree
 * @param                                  $user
 * @param                                  $iscurrentuser
 * @param                                  $course
 */
function theme_ned_boost_myprofile_navigation(core_user\output\myprofile\tree $tree, $user, $iscurrentuser, $course) {
    NED::myprofile_navigation($tree, $user, $iscurrentuser, $course);
}

/**
 * Callback function for theme_boost_union to allow NED Boost to add cards to the Boost Union settings overview page.
 * This function is expected to return an array of arrays containing values with the keys 'label', 'desc', 'btn' and 'url'.
 *
 * @return array
 */
function theme_ned_boost_extend_busettingsoverview() {

    $cards[] = [
        'label' => get_string('pluginname', 'theme_ned_boost'),
        'desc' => get_string('settingsoverview_buc_desc', 'theme_ned_boost'),
        'btn' => 'primary',
        'url' => new \moodle_url('/admin/settings.php', ['section' => 'theme_ned_boost']),
    ];

    return $cards;
}
