<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * @package    theme_ned_classic
 * @subpackage NED Classic
 * @copyright  2018 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @developer  G J Barnard - {@link http://about.me/gjbarnard} and
 *                           {@link http://moodle.org/user/profile.php?id=442195}
 */

// This line protects the file from being accessed by a URL directly.
defined('MOODLE_INTERNAL') || die();

use theme_ned_classic\shared_lib as NED;

include_once("$CFG->dirroot/user/classes/output/myprofile/manager.php");
include_once("$CFG->dirroot/user/profile/lib.php");
include_once("$CFG->dirroot/user/profile/definelib.php");
include_once("$CFG->dirroot/mod/url/locallib.php");
include_once("$CFG->dirroot/lib/weblib.php");
include_once("$CFG->dirroot/local/ned_controller/lib.php");

/**
 * Inject SCSS content.
 *
 * Note: result of this function is saved in theme cache,
 *  so after changes of function work you need purging the caches or enabling theme designer mode
 *
 * @param theme_config $theme The theme config object.
 *
 * @return string
 */
function theme_ned_classic_get_main_scss_content($theme) {
    // return scss from the parent theme
    return theme_classic_get_main_scss_content($theme);
}

/**
 * Get SCSS to prepend.
 * Return the SCSS to prepend to main SCSS for this theme
 * @see theme_config::get_css_content_from_scss() - called from here
 *
 * Note: result of this function is saved in theme cache,
 *  so after changes of function work you need purging the caches or enabling theme designer mode
 *
 * @param theme_config $theme The theme config object.
 *
 * @return string The custom pre SCSS.
 */
function theme_ned_classic_get_pre_scss($theme) {
    return local_ned_controller_get_pre_scss($theme);
}

/**
 * Inject additional SCSS.
 *
 * @param theme_config $theme The theme config object.
 * @return string
 */
function theme_ned_classic_get_extra_scss($theme) {
    return local_ned_controller_get_extra_scss($theme);
}

/**
 * Get compiled css.
 *
 * @return string compiled css
 */
function theme_ned_classic_get_precompiled_css() {
    //global $CFG;
    //return file_get_contents($CFG->dirroot . '/theme/ned_classic/style/moodle.css');
    return '';
}

/**
 * Serves any files associated with the theme settings.
 *
 * @param stdClass $course
 * @param stdClass $cm
 * @param context $context
 * @param string $filearea
 * @param array $args
 * @param bool $forcedownload
 * @param array $options
 * @return bool
 */
function theme_ned_classic_pluginfile($course, $cm, $context, $filearea, $args, $forcedownload, array $options = array()) {
    if ($context->contextlevel == CONTEXT_SYSTEM and ($filearea === 'logo' || $filearea === 'smalllogo')) {
        static $theme = null;
        if ($theme === null) {
            $theme = theme_config::load(NED::$PLUGIN);
        }
        // By default, theme files must be cache-able by both browsers and proxies.
        if (!array_key_exists('cacheability', $options)) {
            $options['cacheability'] = 'public';
        }
        return $theme->setting_file_serve($filearea, $args, $forcedownload, $options) ?: false;
    } else {
        send_file_not_found();
    }
    return false;
}

/**
 * Returns an object containing HTML for the areas affected by settings.
 *
 * Do not add Classic specific logic in here, child themes should be able to
 * rely on that function just by declaring settings with similar names.
 *
 * @param renderer_base $output Pass in $OUTPUT.
 * @param moodle_page $page Pass in $PAGE.
 * @return stdClass An object with the following properties:
 *      - navbarclass A CSS class to use on the navbar. By default ''.
 *      - heading HTML to use for the heading. A logo if one is selected or the default heading.
 *      - footnote HTML to use as a footnote. By default ''.
 */
function theme_ned_classic_get_html_for_settings(renderer_base $output, moodle_page $page) {
    global $CFG;
    $return = new stdClass;

    $return->navbarclass = '';
    if (!empty($page->theme->settings->invert)) {
        $return->navbarclass .= ' navbar-inverse';
    }

    // Only display the logo on the front page and login page, if one is defined.
    if (!empty($page->theme->settings->logo) &&
        ($page->pagelayout == 'frontpage' || $page->pagelayout == 'login')) {
        $return->heading = html_writer::div('', ['class' => 'logo']);
    } else {
        $return->heading = $output->page_heading();
    }

    $return->footnote = '';
    if (!empty($page->theme->settings->footnote)) {
        $return->footnote = \html_writer::div(format_text($page->theme->settings->footnote), 'footnote text-center');
    }

    return $return;
}

/**
 * @param string $option_name
 */
function theme_ned_classic_change_videofield($option_name=''){
    NED::change_videofield($option_name);
}

/**
 * @param \core_user\output\myprofile\tree $tree
 * @param                                  $user
 * @param                                  $iscurrentuser
 * @param                                  $course
 */
function theme_ned_classic_myprofile_navigation(core_user\output\myprofile\tree $tree, $user, $iscurrentuser, $course) {
    NED::myprofile_navigation($tree, $user, $iscurrentuser, $course);
}

/**
 * Let the theme initialise the page object (usually $PAGE).
 *
 * This may be used for example to request jQuery in add-ons.
 *
 * @param moodle_page $page
 */
function theme_ned_classic_page_init(moodle_page $page){
    global $DB, $USER;
    if ($page->context->contextlevel == CONTEXT_MODULE && $page->cm->modname === 'assign') {
        if ($DB->record_exists_sql("SELECT pc.* FROM {assign_plugin_config} pc
                                     WHERE pc.plugin = 'proxy' AND  pc.assignment = ?
                                       AND pc.subtype = 'assignsubmission' AND pc.name = 'enabled'
                                       AND pc.value = '1'", [$page->activityrecord->id])) {
            $paths = explode('/', $page->context->path);
            array_pop($paths);
            $paths = implode('/', $paths);
            if (isset($USER->access['ra'][$paths][5])) {
                $submission = $DB->get_record('assign_submission', [
                    'assignment' => $page->activityrecord->id,
                    'userid' => $USER->id,
                    'latest' => 1
                ]);
                if ($submission && $submission->status === 'submitted') {
                    $page->requires->js_amd_inline("
                        require(['jquery'], function($) {
                            var proxy = $(\"div[class*='summary_assignsubmission_proxy_']\");
                            proxy.parent().parent().remove();
                            var feedback = $(\"div[class*='summary_assignfeedback_editpdf']\");
                            feedback.parent().parent().remove();
                        });");
                }
            }
        }
    }
}
